#include "inc/hprf_androidIntfc.h"
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/spi/spi.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/earlysuspend.h>
#include <linux/firmware.h>
#include <linux/regulator/consumer.h>
#include <linux/device.h>
#include <linux/pm.h>
#include <asm/uaccess.h>
#include <linux/fs.h>
#include <linux/kfifo.h>
#include <linux/sched.h>
#include <linux/kthread.h>
#include <mach/iomux-mx53.h> // For io-pin cfg during suspend/resume

#include "smac_app.h"
#include "smac_drivers.h"
#include "smac_mcu_spi_config.h"
#include "hprfgw_config.h"
#include "smac_802_15_4_shim.h"
#include "hprfgw_rfSharedDefs.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfDiagnostics.h"
#include "eep_params.h"
#include "m24m01.h"
#include "ERInterfaceDLL/nvram_dataMap.h"

#define DRIVER_NAME "mc13192"

#define RFREC_TIMER_RATE_MSEC   10
#define RF_LQI_INTERVAL_MS                          1000        // Task interval in ms
#define HPRF_LQI_RUN_EVENT 0x00000001

//TODO: Queue size arbitrarily chosen
#define USERSPACE_QUEUE_SIZE (256*100)

#define MC13192_MAX_DEVICES 3
#ifdef MC_USE_CDEV
static dev_t mc13192_devno;
static struct class *mc13192_class;
#endif

/**
 * TODO: Remove this crap that exists in the upper layers that is needed if
 * we're not building that.
 */
//void RFReceive_Timer_HISR_Function() {}
//UINT8 gPowerBoost;
//const TYPE_INTERFACE_TO_RFAPP_APP RFIntRFApp;

u8 reg_arr[] = {
	0x06,
	0x08,
	0x11,
	0x2c,
};

const UINT8 bogus_uid[UID_MAX_SIZE] = {
//	0x60, 0x55, 0x0f, 0xf7,
//	0x19, 0xb5, 0x43, 0x67,
    0x00, 0x10, 0x7f, 0x00,
    0x00, 0x05, 0x13, 0x34
};

const UINT8 bogus_bootid[] = "TST-600 [v1.0.1234.56789 #00000000]";
//TODO: Remove bogus version.
const char *bogus_smac_version = "MAC " "SMAC " " Ver " "4.20" " Build: " "Feb 09 2014" " " "12:21:09";
// 0xcc22 is ack

char bogus_broadcastUid[UID_MAX_SIZE] = {
    0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff
};

char bogus_gatewayUid[UID_MAX_SIZE] = {
    0x00, 0x10, 0x7f, 0x00,
    0x01, 0x00, 0x27, 0x3e
};

/* very dirty, function defined in dm_linux_internal.c that is needed here. */
int fifo_len_spinlocked(struct kfifo_rec_ptr_2 *fifo, spinlock_t *slock);



/////////////////////////////////////////////////////////////////////////////////
// Control code and structures required for the RF diagnostics.
#define HPR_DIAG_CTRL_NONE              0
#define HPR_DIAG_CTRL_SHOW_DIAG_INFO    1
#define HPR_DIAG_CTRL_PHY_INFO          2
#define HPR_DIAG_CTRL_DESCR             3
#define HPR_DIAG_CTRL_RF_STACK_TRACE    4
#define HPR_DIAG_CTRL_RF_STACK_DIAG     5
#define HPR_DIAG_CTRL_RF_CLR_DIAG       6
#define HPR_DIAG_CTRL_RF_PHY_READ       7
#define HPR_DIAG_CTRL_RF_ERR_LOG        8
#define HPR_DIAG_CTRL_RF_NETLINK        9
#define HPR_DIAG_CTRL_RF_VERSION        10
#define HPR_DIAG_CTRL_RF_POWER_DBM      11

/*--------------------------------------------------
 * Function:        ConvertTimeToHMS
 *
 * Returns:         char *
 *
 * Return Values:   pointer to destination string
 *
 * Description:     This routine converts the time stamp to days hh:mm:ss.ss format
 *
 * Parameters:      unsigned long Timestamp - number of ticks since bootup in 1/100 s
 *                  char *DestString - pointer to destination
 * Reentrant
 * --------------------------------------------------*/
#define TNUM_DAYS     (24*60*60*1000)
#define TNUM_HOURS    (60*60*1000)
#define TNUM_MINS     (60*1000)
#define TNUM_SECS     (1000)

char *ConvertTimeToHMS(unsigned long Timestamp, char *DestString)
{
   int Days, Hours, Mins, Secs;

   Days = Timestamp / TNUM_DAYS;
   Timestamp = Timestamp % TNUM_DAYS;
   Hours = Timestamp / TNUM_HOURS;
   Timestamp = Timestamp % TNUM_HOURS;
   Mins = Timestamp / TNUM_MINS;
   Timestamp = Timestamp % TNUM_MINS;
   Secs = Timestamp / TNUM_SECS;
   Timestamp = Timestamp % TNUM_SECS;

   sprintf(DestString, "%d days %02d:%02d:%02d.%02lu", Days, Hours, Mins, Secs, Timestamp);
   return(DestString);
}

//UINT32 RF_Diagnostics(void* pBufIn, UINT32 dwLenIn, void* pBufOut, UINT32 dwLenOut);

//  RF Device internal header, set when posting a packet to the rf transmit task
//typedef struct  {
//    UINT8   type;                               //  primary type
//    UINT8   subtype;                            //  sub-type
//    UINT8   req;                                //  requester
//    UINT8   hdroptions1;                        //  frame transmit options (ack, noack, broadcast, unicast, priority etc...)
//    UINT8   hdroptions2;                        //  unused
//    UINT16  unused;                             //  unused
//    TYPE_RFSTACK_CB cb;                         //  call back function
//}   TYPE_RF_DEVICE_RFTX_INTERNAL_HDR;

//typedef struct  {
//    //  Queued message with header and data
//    TYPE_RF_DEVICE_RFTX_INTERNAL_HDR inthdr;    //  Internal RF packet header
//    UINT8   data[RF_DEVICE_DATA_HOLD_BUFFER_SIZE];
//
//    //  Message in process parameters
//    UINT32  len;                                //  data hold buffer length
//    UINT16  index;                              //  index into the data hold buffer
//    UINT8   totalsegment;                       //  total segments of the data hold buffer
//    UINT8   segmentno;                          //  segment number of the data hold buffer
//}   TYPE_RF_DEVICE_RF_DATA_HOLD_BUF;

//typedef struct  {
//    UINT8   flags;
//    UINT8   linkcondition;
//    UINT8   deviceoptions;
//    UINT16  randomno;
//    UINT8   power;
//    UINT8   rfid;
//    UINT32  tsid;
//    UINT16  panid;
//    UINT8   idstring[RF_DEVICE_MAX_CRESNET_ID_STRING_LEN];
//}   TYPE_RF_LINK_REQUEST_PACKET_DATA;

//char CEData[] = {
//    53 21 cc b4 ff ff ff ff  S!......
//    ff ff ff ff ff ff ed fe  ........
//    00 10 7f 00 00 00 51 33  ......Q3
//    ff 7d 48 53 35 14 01 01  .}HS5... //Link quality, sequence, signature
//    00 01 01 11 c0 29 c4 01  .....)..
//    03 00 00 00 00 fe ed 54  .......T
//    53 54 2d 36 30 30 20 5b  ST-600.[
//    76 31 2e 30 2e 34 38 39  v1.0.489
//    34 2e 33 34 38 36 38 20  4.34868.
//    23 30 30 30 30 30 30 30  #0000000
//    30 5d 0d d7 00 85 11     0].....
//};

char broadcastData[] = {
    0xff, 0x01, /* zigbee network */
    0x48, 0x53, 0x35, 0x14, 0x01, 0x01, /* 0x4853 signature, Application support */
    0x00, /* Application framework - Frame Type*/
    0x01, 0x01, 0x11, /* Cluster Library */
    0xcc, 0x7f, /* Application payload */
    0x00, 0x00,                                    /* Panel name starts on this byte. */
    0x03, 0x00, 0x00, 0x28, 0x39, 0xfe, 0xed, 0x54,
    0x53, 0x54, 0x2d, 0x36, 0x30, 0x30, 0x20, 0x5b,
    0x76, 0x31, 0x2e, 0x30, 0x2e, 0x34, 0x38, 0x39,
    0x34, 0x2e, 0x33, 0x34, 0x38, 0x36, 0x38, 0x20,
    0x23, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x5d,
//    0x0c, 0xba, 0x00, 0x8b, 0x91 /* Radio info */
};

Rf_802_15_MAC_TO_SMAC_Frame_struct testPacket = {
	.msduHandle = 1,
	.RfData = {
		.header = {
		    // "ACK Required, NOT intrapan" bit
			.frameCtl = 0xcc23, //16-bit
			.sequenceNo = 1,
			.dstPanId = { 0xff, 0xff },
			//.dstAddr = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
			.srcPanId = { 0xfe, 0xed },
//			.srcAddr = {
//				0x00, 0x10, 0x7f, 0x00,
//				0x00, 0x05, 0x13, 0x34
//			},
		},
//		.data = {
//		    'I', '\'','m', ' ', 'a', ' ', 'l', 'i',
//		    't', 't', 'l', 'e', ' ', 'p', 'a', 'c',
//		    'k', 'e', 't', ',', ' ', 's', 'h', 'o',
//		    'r', 't', ' ', 'a', 'n', 'd', ' ', 's',
//		    't', 'o', 'u', 't'
//		},
	},
};

BOOL readWriteEeprom(EEP_PARAMS *eep_param, BOOL doRead);

/**
 * \brief       Gets the power table index based on the RF
 *              module and the table present in the file
 *              hprfgw_rfrec.c:
 *              RFTransmitPowerValuesInDbm[][]
 *              RFPowerLmts[][]
 * \retval      Returns the index to the power table
 */
UINT8 GetRFPowerTableIndex(void)
{
    /* TODO: Set power table default back to international. */
    UINT8 index = MAX_POWER_LIMIT_TABLE;//DEFAULT_POWER_LIMIT_TABLE;
    EEP_PARAMS eep_param;
    BOOL bReadOK;

    //RETAILMSG(1, (TEXT("GetRFPowerTableIndex\r\n")));
    bReadOK = readWriteEeprom(&eep_param, 1);      // read the ER parameters from the data store

    if(bReadOK) {
        // See if this is a valid table
        if((eep_param.RfPowerTable & EEP_PT_SIG_MASK) == EEP_PT_SIGNATURE) {
            // Yes, use power table index
            index = eep_param.RfPowerTable & ~EEP_PT_SIG_MASK;
        }
        else {
            /* TODO: Get from the registry key? */
            RETAILMSG(1, (TEXT("GetRFPowerTableIndex: power table signature check failed.\r\n")));
        }
    }
    else {
        RETAILMSG(1, (TEXT("GetRFPowerTableIndex: failed to open critical store.\r\n")));
    }

    if(index > MAX_POWER_LIMIT_TABLE)
        index = MAX_POWER_LIMIT_TABLE;//DEFAULT_POWER_LIMIT_TABLE;

    //RETAILMSG(1, (TEXT("GetRFPowerTableIndex: returned index = 0x%x\r\n"),index));
    return (index);
}

// Constants for LQI handler
#define RF_LQI_TASK_PRIORITY                        (RF_TASK_PRIORITY(RF_TASKS_BASE_PRIORITY,15))
#define RF_LQI_INTERVAL_MS                          1000        // Task interval in ms

#pragma pack(push, 1)
struct lqi_message
{
    u8 messageType;
    u32 lq_norm_local_device;
    s32 local_dev_dbm;
    u32 lq_norm_gateway;
    s32 remote_dbm;
    u8 padding[3]; // Pad to 20 bytes.
};
#pragma pack(pop)

void lqi_response_to_app(struct mc13192 *rf)
{
    struct lqi_message app_msg;
    UINT16 lq_norm_local_device, lq_norm_gateway;
    // LQ at the local device (panel) and gateway
    lq_norm_local_device = RFIntRFApp.pF_ConvertLQIToIEEEScaleWord(rf->localDevLqi, LQI_DISP_OFFSET);
    lq_norm_gateway = RFIntRFApp.pF_ConvertLQIToIEEEScaleWord(rf->remoteLqi, LQI_DISP_OFFSET);

    app_msg.messageType = HPRF_CTRL_MSG_LQI_DATA;
    app_msg.lq_norm_local_device = lq_norm_local_device;
    app_msg.local_dev_dbm = RFIntRFApp.pF_ConvertLQITodBm(rf->localDevLqi);
    app_msg.lq_norm_gateway = lq_norm_gateway;
    app_msg.remote_dbm = RFIntRFApp.pF_ConvertLQITodBm(rf->remoteLqi);

    // Post Control message to the application
    SendCtrlMsgToApp(&app_msg, sizeof(struct lqi_message));
}

INT32 RF_LQI_Timer_EchoRequest_CB(TYPE_RF_ECHO_REPLY_PACKET *pechoreply)
{
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);
    rf->localDevLqi = pechoreply->localrxlqi;
    rf->remoteLqi   = pechoreply->remoterxlqi;
    return (SUCCESS);
}

INT32 RF_LQI_Timer_NetworkService_CB(TYPE_RF_DISCOVER_REPLY *pRFDis, BOOL ignore)
{
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);
    rf->localDevLqi = pRFDis->linkqualityatlocaldevice;
    rf->remoteLqi   = pRFDis->disc.linkquality;
    return (SUCCESS);
}

void lqi_send_discover(struct mc13192 *rf)
{
    static TYPE_RF_DISCOVER_REQUEST Rf_Serv_adreq = {
            .firstchannel = 0,
            .lastchannel = 0,
            .service      = RF_PACKET_DISCOVER_NETDEV_REQUEST,
            .detailview = FALSE,
            .scan = TRUE,
            .flags = RF_DISCOVER_GATEWAYS,
            .waitforto = FALSE,
            .pcbf = RF_LQI_Timer_NetworkService_CB,
    };
    /* Clear the value ( = 255 in dBm) before posting the next message */
    rf->localDevLqi = 255;  /* For the echo, it will be cleared automatically */
    rf->remoteLqi   = 255;
    /* Channel is the only part that updates on subsequent calls. */
    Rf_Serv_adreq.firstchannel = rf->rfparam.channel;
    Rf_Serv_adreq.lastchannel = rf->rfparam.channel;
    RFIntRFApp.pF_NetworkService (&Rf_Serv_adreq);
}

void lqi_send_echoreq(struct mc13192 *rf)
{
    static TYPE_RF_ECHO_REQUEST_PACKET Rf_Serv_Echoreq = {
            .rfid = 0,
            .repeats = 1,
            .buf = "TESTLQI",
            .buflen = 8,
            .start = TRUE,
            .pcbf = RF_LQI_Timer_EchoRequest_CB
    };
    RFIntRFApp.pF_PostEchoPacketRequest(&Rf_Serv_Echoreq);
}

int rf_lqi_thread(void *pContext)
{
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);
    UINT32 eventResult;
    int err;

    struct sched_param lqi_prio;
    lqi_prio.sched_priority = RF_LQI_TASK_PRIORITY;
    err = sched_setscheduler(rf->lqiThread, SCHED_RR, &lqi_prio);
    if(err) {
        pr_err("=========== Failed to set LQI priority to %i =========== task = %i name = %s err = %i\n",
                lqi_prio.sched_priority,
                current->pid,
                "LQI",
                err);
    }

    RFIntHostApp.pF_StartupWait(HPRF_STARTUP_ALL_READY);
    RFIntHostApp.pF_ErrorPrintf("\nRF LQI packet thread: running!");
    while(1) {
        eventResult = OsWaitEventGroupAll(rf->lqiEvent, HPRF_LQI_RUN_EVENT);
        if(eventResult) {
            lqi_response_to_app(rf);
            if (RFIntRFApp.pF_GatewayConnectionStatusQuery() == TRUE) {
                lqi_send_echoreq(rf);
            }
            else {
                lqi_send_discover(rf);
            }
        }
        else {
            DmConsolePrintf("RF_LQIThread: Failed Wait Object. Result: %lu\n", eventResult);
        }
        msleep(RF_LQI_INTERVAL_MS);
    }
    return 0;
}

int writeToUserspaceQueue(struct mc13192 *rf, struct mc13192_pipe_dev *pipe,
        char *buf, int bufSize, UINT32 timeout, bool force)
{
    int bytes = 0;
    if(rf->isAppInitDone || force) {
        bytes = kfifo_in_spinlocked(&pipe->fifo, buf, bufSize, &pipe->slock);
        if(!bytes) {
            pr_err("HPRF writeToUserspaceQueue Failed.\n");
        }
        wake_up_interruptible(&pipe->wq);
    }
    return bytes;
}

// Send Control message to Application
UINT32 SendCtrlMsgToApp(void *lpBuffer, UINT32 cbDataSize)
{
    BOOL bWriteMsg = FALSE;
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    //TODO: make more like CE version and not block at all.
    //dev_info(&spi->dev, "%s - sending %lu bytes to ctrl queue\n", __func__, cbDataSize);

    if(!rf->hack_isSleeping) {
        bWriteMsg = writeToUserspaceQueue(rf, &rf->ctl_pipe, lpBuffer, cbDataSize, OS_NO_WAIT, false);
    }
    return (UINT32)bWriteMsg;
#if 0
    // 20100518: Check for power state D3. We send the GW connect message in PowerUp function and then
    // we see GW and CTRL system disconnect message. If it reaches application, it seems to discard
    // the initial key press from sending to control system. Adding this check prevents that by not sending
    // disconnected messages to application when we are coming out of suspend mode.
    if (pHPRDevice->RF_Cntrl_Queue_write && pHPRDevice->bApplicationInitDone && pHPRDevice->powerState < D3)
    {
        bWriteMsg = WriteMsgQueue(pHPRDevice->RF_Cntrl_Queue_write, lpBuffer, cbDataSize, OS_NO_SUSPEND, 0);
        if (bWriteMsg == FALSE)
            RETAILMSG(1, (TEXT("HPRF SendCtrlMsgToApp Fail. Error: %d\r\n"), GetLastError()));
    }
    else
    {
        if (pHPRDevice->RF_Cntrl_Queue_write == NULL)
            RETAILMSG(1, (L"HPR SendCtrlMsgToApp: Invalid handle\r\n"));
    }
#endif
}

// Send Cresent packet to Application
UINT32 SendCresnetPacketToApp(void *lpBuffer, UINT32 cbDataSize)
{
    BOOL bWriteMsg = FALSE;
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    bWriteMsg = writeToUserspaceQueue(rf, &rf->data_pipe, lpBuffer, cbDataSize, 100, false);
    return (UINT32)bWriteMsg;
}


// Send Local Cresent packet to Application
UINT32 SendLocalCresnetPacketToApp(void *lpBuffer, UINT32 cbDataSize)
{
    BOOL bWriteMsg = FALSE;
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    bWriteMsg = writeToUserspaceQueue(rf, &rf->data_pipe, lpBuffer, cbDataSize, 100, false);
    return (UINT32)bWriteMsg;
}

// Send Diagnostics and test data messages to Application
UINT32 SendDiagDataToApp(void *lpBuffer, UINT32 cbDataSize)
{
    BOOL bWriteMsg = 0;
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    bWriteMsg = writeToUserspaceQueue(rf, &rf->diag_pipe, lpBuffer, cbDataSize, OS_NO_WAIT, false);
    return (UINT32)bWriteMsg;
}

//TODO: Remove or put into use.
static int printAsHexChars(char *outBuf, size_t outBufSize, void *inBuf, size_t inBufSize)
{
    int i;
    size_t bytes = 0;
    char *cInBuf = (char*)inBuf;
    for(i=0; i < inBufSize; ++i) {
        bytes += scnprintf(&outBuf[bytes], outBufSize-bytes, "%02hhx", cInBuf[i]);
    }
    return bytes;
}

static int scanHexChars(const char *inBuf, size_t inSize, void *outBuf, size_t outBufSize)
{
    size_t elements = 0;
    int res;
    size_t i;
    size_t nCopy;
    unsigned char *cOutBuf = (unsigned char*)outBuf;
    /* inSize should be double the outBufSize. */
    nCopy = (inSize < outBufSize*2) ? inSize/2 : outBufSize;
    pr_info("inSize = %u outSize = %u nCopy = %u\n", inSize, outBufSize, nCopy);
//    nCopy = nCopy & (~1ul);
    for(i=0; i < nCopy; ++i) {
        res = sscanf(&inBuf[i*2], "%02hhx", &cOutBuf[i]);
        if(!res) break;
        elements += res;
    }
    return elements;
}

/** User interface for RF device configuration */
static ssize_t rfid_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 4, "%02hhx\n", rf->rfparam.RFID);
}

static ssize_t rfid_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhx", &rf->rfparam.RFID);

    /* Send a special rfid packet up to cipclient so it knows the driver's RFID changed. */
    writeToUserspaceQueue(rf, &rf->data_pipe, &rf->rfparam.RFID, 1, OS_WAIT_FOREVER, true);
    return size;
}

static ssize_t uid_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 18, "%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx\n",
            rf->rfparam.UID[0], rf->rfparam.UID[1],
            rf->rfparam.UID[2], rf->rfparam.UID[3],
            rf->rfparam.UID[4], rf->rfparam.UID[5],
            rf->rfparam.UID[6], rf->rfparam.UID[7]);
}

static ssize_t uid_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx",
            &rf->rfparam.UID[0], &rf->rfparam.UID[1],
            &rf->rfparam.UID[2], &rf->rfparam.UID[3],
            &rf->rfparam.UID[4], &rf->rfparam.UID[5],
            &rf->rfparam.UID[6], &rf->rfparam.UID[7]);

    RFIntRFApp.pF_SetLocaldeviceUID_PANID(TRUE, &rf->rfparam.UID[0], rf->rfparam.panID);
    //TODO: Need to write eeprom.
    rf->rfparamlocal.UID_Valid = UID_SIGN;
    return size;
}

static ssize_t panid_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 6, "%04hx\n", rf->rfparam.panID);
}

static ssize_t panid_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hx", &rf->rfparam.panID);
    RFIntRFApp.pF_SetLocaldeviceUID_PANID(TRUE, &rf->rfparam.UID[0], rf->rfparam.panID);
    return size;
}

static ssize_t gw_panid_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 6, "%04hx\n", rf->rfparam.gatewayPanID);
}

static ssize_t gw_panid_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hx", &rf->rfparam.gatewayPanID);
    RFIntRFApp.pF_SetGatewayUID_PANID_ADDRMODE(TRUE, &rf->rfparam.gatewayUID[0], rf->rfparam.gatewayPanID, rf->rfparam.addrMode);
    return size;
}

static ssize_t gw_uid_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 18, "%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx\n",
            rf->rfparam.gatewayUID[0], rf->rfparam.gatewayUID[1],
            rf->rfparam.gatewayUID[2], rf->rfparam.gatewayUID[3],
            rf->rfparam.gatewayUID[4], rf->rfparam.gatewayUID[5],
            rf->rfparam.gatewayUID[6], rf->rfparam.gatewayUID[7]);
}

static ssize_t gw_uid_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx",
            &rf->rfparam.gatewayUID[0], &rf->rfparam.gatewayUID[1],
            &rf->rfparam.gatewayUID[2], &rf->rfparam.gatewayUID[3],
            &rf->rfparam.gatewayUID[4], &rf->rfparam.gatewayUID[5],
            &rf->rfparam.gatewayUID[6], &rf->rfparam.gatewayUID[7]);

    RFIntRFApp.pF_SetGatewayUID_PANID_ADDRMODE(TRUE, &rf->rfparam.gatewayUID[0], rf->rfparam.gatewayPanID, rf->rfparam.addrMode);
    return size;
}

static ssize_t cca_thresh_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 5, "%hhu\n", rf->rfparam.CCAEnergyDetectThr);
}

static ssize_t cca_thresh_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhu", &rf->rfparam.CCAEnergyDetectThr);
    RFIntRFApp.pF_SetCCAEnergyDetectThr(rf->rfparam.CCAEnergyDetectThr);
    return size;
}

static ssize_t power_comp_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 5, "%hhu\n", rf->rfparam.RSSIPowerCompOffset);
}

static ssize_t power_comp_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhu", &rf->rfparam.RSSIPowerCompOffset);
    RFIntRFApp.pF_SetRSSIPowerCompOffset(rf->rfparam.RSSIPowerCompOffset);
    return size;
}

static ssize_t channel_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 4, "%hhu\n", rf->rfparam.channel);
}

static ssize_t channel_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhu", &rf->rfparam.channel);
    RFIntRFApp.pF_SetChannel(rf->rfparam.channel);
    return size;
}

static ssize_t retry_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 4, "%hhu\n", rf->rfparam.retry);
}

static ssize_t retry_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhu", &rf->rfparam.retry);
    RFIntRFApp.pF_SetRetryCount(rf->rfparam.retry);
    return size;
}

static ssize_t power_hilo_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 4, "%hhu\n", rf->rfparam.powerhilo);
}

static ssize_t power_hilo_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhu", &rf->rfparam.powerhilo);
    RFIntRFApp.pF_SetPowerLevel(rf->rfparam.powerhilo);
    return size;
}

static ssize_t retry_count_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 4, "%hhu\n", rf->rfparam.txPktretryonconnection);
}

static ssize_t retry_count_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhu", &rf->rfparam.txPktretryonconnection);
    RFIntRFApp.pF_SetRetryCountOnConnectionReq(rf->rfparam.txPktretryonconnection);
    return size;
}

static ssize_t prelink_pkt_count_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 4, "%hhu\n", rf->rfparam.PreLinkQueryPacketCnt);
}

static ssize_t prelink_pkt_count_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhu", &rf->rfparam.PreLinkQueryPacketCnt);
    RFIntRFApp.pF_SetPreLinkQueryPacketCnt(rf->rfparam.PreLinkQueryPacketCnt);
    return size;
}

static ssize_t power_boost_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 4, "%hhu\n", rf->rfparam.powerBoost);
}

static ssize_t power_boost_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    sscanf(buf, "%hhu", &rf->rfparam.powerBoost);
    if(rf->rfparam.powerBoost > 21) {
        rf->rfparam.powerBoost = 21;
    }
    if(RFIntRFApp.pF_RFReceive_GetLocalDevicePower() >= RF_POWER_HI) {
        RFIntRFApp.pF_SetPowerLevel(RF_POWER_HI);
    }
    else {
        RFIntRFApp.pF_SetPowerLevel(RF_POWER_LO);
    }
    return size;
}

static ssize_t tsid_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    //size_t bytes = printAsHexChars(buf, PAGE_SIZE, &rf->rfparamlocal.TSID, sizeof(rf->rfparamlocal.TSID));
    size_t bytes = scnprintf(buf, PAGE_SIZE, "%lx\n", rf->rfparamlocal.TSID);
    return bytes;
}

static ssize_t tsid_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    /* TODO: Don't know if this assumes an okay byte order... */
    sscanf(buf, "%lx", &rf->rfparamlocal.TSID);
    return size;
}

static ssize_t bootid_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, MAX_BOOT_ID_MESSAGE, "%s\n", rf->bootIdMsg);
}

static ssize_t bootid_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    strncpy(rf->bootIdMsg, buf, MAX_BOOT_ID_MESSAGE);
    return size;
}


/* Action or status functions */
static ssize_t mc13192_acquire(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    char *end;
    UINT32 timeout_minutes = simple_strtoul(buf, &end, 10);
    if(timeout_minutes) {
        dev_info(dev, "acquire_start - %lu minutes\n", timeout_minutes);
        RFIntRFApp.pF_AcquireStart(timeout_minutes);
    }
    else {
        dev_info(dev, "acquire_stop\n");
        RFIntRFApp.pF_AcquireStop(1);
    }
    return size;
}
static ssize_t mc13192_connect(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    dev_info(dev, "lookup settings\n");
    RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();
    return size;
}

static ssize_t init_done_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    rf->isAppInitDone = TRUE;
    if (rf->rfparamlocal.UID_Valid == UID_SIGN)
    {
        RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();
    }
    return size;
}

static ssize_t signal_strength_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    UINT8 lq = 0;
    UINT16 lq_norm = 0;
    if(!rf->lqiActive) {
        RFIntRFApp.pF_GetLQI(0, TRUE, &lq);
        lq_norm = RFIntRFApp.pF_ConvertLQIToIEEEScaleWord(lq, LQI_DISP_OFFSET);
    }
    return scnprintf(buf, 7, "%hu\n", lq_norm);
}

static ssize_t lqi_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, 4, "%u\n", rf->lqiActive);
}

static ssize_t lqi_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    int turnOnLqi;
    sscanf(buf, "%u", &turnOnLqi);
    if(turnOnLqi) {
        if(!rf->lqiActive) {
            rf->localDevLqi = 255;
            rf->remoteLqi = 255;
            rf->lqiActive = TRUE;
            OsSetEvent(rf->lqiEvent, HPRF_LQI_RUN_EVENT);
        }
    }
    else {
        if(rf->lqiActive) {
            OsClearEvent(rf->lqiEvent, HPRF_LQI_RUN_EVENT);
            rf->lqiActive = FALSE;
        }
    }
    return size;
}

static ssize_t gw_connected_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t gw_connected = RFIntRFApp.pF_GatewayConnectionStatusQuery();
    return scnprintf(buf, 4, "%u\n", gw_connected);
}

static ssize_t cs_connected_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t cs_connected = RFIntRFApp.pF_ControlSystemConnectionStatusQuery();
    return scnprintf(buf, 4, "%u\n", cs_connected);
}

static ssize_t gw_acquired_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    size_t i;
    uint8_t gw_acquired;

    for(i = 0; i < UID_MAX_SIZE; ++i) {
        if(rf->rfparam.gatewayUID[i] != 0) {
            break;
        }
    }
    gw_acquired = (i == UID_MAX_SIZE) ? 0 : 1;
    return scnprintf(buf, 4, "%u\n", gw_acquired);
}

static ssize_t gw_name_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    return scnprintf(buf, GATEWAY_NAME_MAX_SIZE, "%s\n", rf->rfparamlocal.gatewayName);
}

static ssize_t diag_show_info_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    ssize_t bytes = 0;
    size_t size = PAGE_SIZE;

    UINT8 *pindex;
    UINT8 indexArray[MAX_DESCRIPTIVE_TABLE_ENTRIES];
    TYPE_RF_DESCRIPTIVE_TABLE *pDesct;
    int count = RFDesc_GetIndexToDescriptiveTableAllDevices (indexArray);
    pindex = indexArray;
    while (count-- && (*pindex < g_MaxSupportedRFDevices))
    {
        pDesct = RFDesc_GetDescriptiveEntryIDInfo(*pindex);
        pindex++;
    }
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", pDesct->linkstatecnt);
    bytes += scnprintf(&buf[bytes], size-bytes, "%hhu ", pDesct->LQ.linkqualityAtLocalDevice);
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", pDesct->COMLog.txpacketcount);
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", pDesct->COMLog.txpacketretrycount);
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", pDesct->COMLog.rxpacketcount);
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", pDesct->COMLog.rxpacketrepeatcount);
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", pDesct->COMLog.rxpacketsequenceErrcount);
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", pDesct->COMLog.rxpacketsegmentationerrcnt);

    // Low level diagnostics (as that in TPS-6X setup screen diag display, sub page)
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumTimer1Ints());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumLOUnLockInts());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumInvalidPhyPacketsRecvd());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumCCAFailures());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelRecvCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelRecvByteCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelTotalLostPackets());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelInvalidRxPacketCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelRetryPacketCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelTxPacketCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelTxPacketBytes());

    // Following are not present in TPS-6X setup screen display
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumCCA_TxDone());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelResets());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumBadTimerISR());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getChannelBusyResets());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumDataConfMessages());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumResetAndNacks());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getCongestion());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getUtilization());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumCCATimeoutsFailures());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumRxInactivityResets());

    // PHY Information
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumTimer1Ints());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu",  Rfdiags_getLowLevelPhyChipId());
    return bytes;
}

static ssize_t diag_desc_dump_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    size_t index = 0;
    uint8_t tmpBuf[100];
    UNSIGNED pipesize, available, messages;

    HPR_TY_DiagData DescrData;
    TYPE_RF_DESCRIPTIVE_TABLE *pDesct;

    sscanf(buf, "%u", &index);
    pDesct = RFDesc_GetDescriptiveEntryIDInfo(index);
    if (pDesct == NULL)
    {
        pr_err("RF_Diagnostics: RF Descriptor NULL\n");
        // Type cast the output buffer
        return -EFAULT;
    }
    // Fill in data and control code and send to app.
    DescrData.dwDataCode = HPRF_DIAG_DATA_DESCR_COMMAND_REPLY;

    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " [RF DESCRIPTOR REPLY]\r\n");
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Descriptive entry:                     %d\r\n", index);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " IDstring:                              %s\r\n",   pDesct->IDInfo.idstring);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " UID:                                   %s\r\n",   ConvertUIDtoString ((UINT8*)&pDesct->IDInfo.uid, tmpBuf, gAddrModeLong_c));
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Panid:                                 %04x\r\n", pDesct->IDInfo.panid);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Addrmode:                              %d\r\n", pDesct->IDInfo.addrmode);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " RFID:                                  %02x\r\n", pDesct->IDInfo.rfid);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " TSID:                                  %08lx\r\n", pDesct->IDInfo.tsid);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Power:                                 %d\r\n", pDesct->IDInfo.power);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Deviceoptions:                         %d\r\n", pDesct->IDInfo.deviceoptions);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link state:                            %s\r\n",   ConvertLinkStatetoString (pDesct->linkstate, tmpBuf));
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link state connection count:           %lu\r\n", pDesct->linkstatecnt);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    //  new
    ConvertTimeToHMS((OS_RetrieveClock() - pDesct->linkeactiveestablishedtime), tmpBuf);
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link active time:                      %lu (%s)\r\n", pDesct->linkeactiveestablishedtime, tmpBuf);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    //  new
    RFTransmit_GetPipeAvailableMemorySizeByIndex ((UINT8)index, &pipesize, &available, &messages);
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Transmit Pipe info size/avaliable/msgs:%lu/%lu/%lu\r\n", pipesize,available,messages);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Linkquality at local device:           %d (%d dBm)\r\n", pDesct->LQ.linkqualityAtLocalDevice, RFReceive_ConvertLinkQualityTodBm(pDesct->LQ.linkqualityAtLocalDevice));
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Linkquality at remote device:          %d (%d dBm)\r\n", pDesct->LQ.linkqualityAtRemoteDevice, RFReceive_ConvertLinkQualityTodBm(pDesct->LQ.linkqualityAtRemoteDevice));
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Waiting for ack flag:                  %d\r\n",   pDesct->waitingforACK);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code available flag:            %d\r\n",   pDesct->resultcodeavailable);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code:                           %d\r\n", pDesct->resultcode);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Retry timer:                           %u\r\n", pDesct->RetryTimer);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Retry timed out:                       %d\r\n", pDesct->RetryTimedOut);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet retry cntr:                  %d\r\n", pDesct->txpacketretrycntr);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Maximum Tx packet retries:             %d\r\n", pDesct->maxtxpacketretries);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Heart beat timer:                      %d\r\n", pDesct->HeartBeatTimer);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Heart beat timed out:                  %d\r\n",   pDesct->HeartBeatTimedOut);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code timer:                     %d\r\n", pDesct->ResultCodeTimer);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code timed out:                 %d\r\n",   pDesct->ResultCodeTimedOut);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet sequence number:             %d\r\n", pDesct->txsequence);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packets:                %lu\r\n", pDesct->COMLog.txpacketcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packet retries:         %lu\r\n", pDesct->COMLog.txpacketretrycount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packet in errors:       %lu\r\n", pDesct->COMLog.txpacketerrorcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of times tx buffer overflow:  %lu\r\n", pDesct->COMLog.txbufoverflowcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet resultcode timeouts:   %lu\r\n", pDesct->COMLog.txpacketresultcodetimeouts);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    // new
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of concatenated tx packet:    %lu\r\n", pDesct->COMLog.txpacketconcatcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of segmented tx packet:       %lu\r\n", pDesct->COMLog.txpacketsegmented);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    //printf(" tx packet                              %u\r\n", pDesct->COMLog.txpacketfailcount);
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet payload data in bytes: %lu\r\n", pDesct->COMLog.txpacketpayloadbytecount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet overhead in bytes:     %lu\r\n", pDesct->COMLog.txpacketoverheadbytecount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    //  modified
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Average tx packet length in bytes:     %lu\r\n", pDesct->COMLog.txpacketcount?
                                                ((pDesct->COMLog.txpacketoverheadbytecount+pDesct->COMLog.txpacketpayloadbytecount)/pDesct->COMLog.txpacketcount):0);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    //  new
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet turn around delay in 10msec: %lu\r\n",    pDesct->COMLog.txpacketturnaroundtime);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet avg turn around delay in 10msec: %lu\r\n",    (pDesct->COMLog.txpacketcount-pDesct->COMLog.txpacketretrycount)?(pDesct->COMLog.txpacketaccturnaroundtime/(pDesct->COMLog.txpacketcount-pDesct->COMLog.txpacketretrycount)):0);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Rx packet sequence number:             %d\r\n",  pDesct->rxsequence);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packets:                %lu\r\n",  pDesct->COMLog.rxpacketcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet repeats:         %lu\r\n",  pDesct->COMLog.rxpacketrepeatcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet sequence err:    %lu\r\n",  pDesct->COMLog.rxpacketsequenceErrcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet segment err:     %lu\r\n",  pDesct->COMLog.rxpacketsegmentationerrcnt);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet with invalid len:%lu\r\n",  pDesct->COMLog.rxpacketlenerror);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet with hdr/footr err:%lu\r\n",    pDesct->COMLog.rxpackethdrftrerror);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of times gw rx buffer overflow: %lu\r\n",  pDesct->COMLog.rxbufoverflowcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    // new
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet concatenation:   %lu\r\n", pDesct->COMLog.rxpacketconcatcount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx acks:                   %lu\r\n", pDesct->COMLog.rxpacketacks);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx naks/cca's:             %lu\r\n", pDesct->COMLog.rxpacketnaks);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total rx packet payload data in bytes: %lu\r\n",    pDesct->COMLog.rxpacketpayloadbytecount);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total rx gw startup packets:           %lu\r\n",    pDesct->COMLog.rxgwrestarts);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of heartbeat timeouts:        %lu\r\n",    pDesct->COMLog.heartbeattimeouts);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no tx cong. avoidance:           %lu\r\n", pDesct->COMLog.txpacketcongavoidance);
    SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

    return size;
}

/* TODO: Can't implement via sysfs. Needs too many entries for the buffer. */
static ssize_t diag_stack_trace_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    ssize_t bytes = 0;
    return bytes;
}

static ssize_t diag_clear_diag(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    pr_info("RF_Diag: Clear RF diag counters.\n");
    //  Clear diagnostic counters
    RFAPP_ClearDiagCounters();
    return size;
}

static ssize_t diag_stack_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    ssize_t bytes = 0;
    size_t size = PAGE_SIZE;

    UINT32 totalResCodeTimeouts = 0;
    UINT8 msgIndex, msgEntries;
    UINT8 errPktCnt = 0;
    size_t index;
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumTimer1Ints());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumLOUnLockInts());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumInvalidPhyPacketsRecvd());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumCCA_TxDone());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelRecvCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelRecvByteCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelTotalLostPackets());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelInvalidRxPacketCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelRetryPacketCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelTxPacketCnt());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelTxPacketBytes());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getLowLevelResets());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumBadTimerISR());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getChannelBusyResets());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumDataConfMessages());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumResetAndNacks());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getCongestion());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getUtilization());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumCCAFailures());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumCCATimeoutsFailures());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", Rfdiags_getNumRxInactivityResets());

    for(index=0; index < g_MaxSupportedRFDevices; ++index) {
        totalResCodeTimeouts += g_pRFDeviceDescriptiveTable[index].COMLog.txpacketresultcodetimeouts;
    }
    RFDesc_GetErrorPktIndex (DEVICE_DESCRIPTIVE_TABLE_INDEX, &msgIndex, &msgEntries, &errPktCnt);
    bytes += scnprintf(&buf[bytes], size-bytes, "%hhu ", errPktCnt);
    return bytes;
}

/* TODO: may need to remain ioctl. */
static ssize_t diag_phy_read_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    ssize_t bytes = 0;
    size_t size = PAGE_SIZE;
    return bytes;
}

/* TODO: may need to remain ioctl. */
static ssize_t diag_err_log_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    ssize_t bytes = 0;
    size_t size = PAGE_SIZE;
    return bytes;
}

static ssize_t diag_netlink_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    ssize_t bytes = 0;
    size_t size = PAGE_SIZE;

    UINT8   IndexArray[MAX_DESCRIPTIVE_TABLE_ENTRIES];
    UINT16 count;
    TYPE_RF_DESCRIPTIVE_TABLE *pDesct;
    UINT8 *pindex;
    char linkStateStr[64]; // Same length as in TYPE_RF_NETLINK_INFO
    char timeStr[64];

    UINT32 connType = 0;

    // Type cast output buffer
//    pRFNetlinkInfo = (TYPE_RF_NETLINK_INFO*)pBufOut;

    // Fill the info..
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", (UINT32)RFIntRFApp.pF_GatewayConnectionStatusQuery());
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", (UINT32)RFIntRFApp.pF_ControlSystemConnectionStatusQuery());

    if (RFReceive_IsControlSystemConnectedNetTypeCresnet()) {
        connType = 1;
    }
    else if (RFReceive_IsControlSystemConnectedNetTypeEthernet()) {
        connType = 2;
    }
    else {
        connType = 0;
    }
    bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", connType);

    IndexArray[0] = DEVICE_DESCRIPTIVE_TABLE_INDEX;
    pindex = IndexArray;
    count = 1;

    RFDesc_SortIndexElementsByRFID (pindex, (UINT8)count);
    pDesct = RFDesc_GetDescriptiveEntryIDInfo(*pindex);
    if (pDesct)
    {
        ConvertLinkStatetoString (g_pRFDeviceDescriptiveTable[*pindex].linkstate, linkStateStr);
        timeStr[0] = 0;
        if (pDesct->linkstate == RF_DEVICE_LINK_STATE_ACTIVE)
        {
            ConvertTimeToHMS((OS_RetrieveClock() - pDesct->linkeactiveestablishedtime), timeStr);
        }
        bytes += scnprintf(&buf[bytes], size-bytes, "(%s) ", linkStateStr);
        bytes += scnprintf(&buf[bytes], size-bytes, "(%s) ", timeStr);

        bytes += scnprintf(&buf[bytes], size-bytes, "%hhu ", g_pRFDeviceDescriptiveTable[*pindex].IDInfo.rfid);
        bytes += scnprintf(&buf[bytes], size-bytes, "%lu ", pDesct->linkstatecnt);
        bytes += scnprintf(&buf[bytes], size-bytes, "%hhu ", RFReceive_ConvertLinkQualityToPercent(pDesct->LQ.linkqualityAtLocalDevice, LQI_DISP_OFFSET));
        bytes += scnprintf(&buf[bytes], size-bytes, "%hhu ", RFReceive_ConvertLinkQualityToPercent(pDesct->LQ.linkqualityAtRemoteDevice, LQI_DISP_OFFSET));
    }
    return bytes;
}

static ssize_t debug_trace_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    UINT32 debugFlags = rf->rfparam.RFTraceFlags;
    return scnprintf(buf, 4, "%lx\n", debugFlags);
}

static ssize_t debug_trace_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct spi_device *spi = to_spi_device(dev);
    struct mc13192 *rf = spi_get_drvdata(spi);

    UINT32 debugFlags;
    sscanf(buf, "%lx", &debugFlags);
    rf->rfparam.RFTraceFlags = debugFlags & (RF_RFTX_DEBUG_FLAG | RF_RFRX_DEBUG_FLAG | RF_GENERIC_DEBUG_FLAG);
    return size;
}

static ssize_t mc13192_show_rfparams(struct device *dev, struct device_attribute *attr, char *buf)
{
    //Bad, we shouldn't be putting so much into the a sysfs file,
    //throwing this in here for convenience, not correctness.
    ssize_t bytes = 0;
    size_t size = PAGE_SIZE;
    int i;
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);
    RF_PARAM *rf_param = &rf->rfparam;
    bytes += scnprintf(&buf[bytes], size-bytes, "UID[");
    for(i=0; i < UID_MAX_SIZE; ++i) {
        bytes += scnprintf(&buf[bytes], size-bytes, "%02x", rf_param->UID[i]);
    }
    bytes += scnprintf(&buf[bytes], size-bytes, "]\n");
    bytes += scnprintf(&buf[bytes], size-bytes, "panId[%04x]\n", rf_param->panID);
    bytes += scnprintf(&buf[bytes], size-bytes, "shortAddr[%04x]\n", rf_param->shortAddr);
    bytes += scnprintf(&buf[bytes], size-bytes, "addrMode [%04x]\n", rf_param->addrMode);
    bytes += scnprintf(&buf[bytes], size-bytes, "CCA[%04x]\n", rf_param->CCAEnergyDetectThr);
    bytes += scnprintf(&buf[bytes], size-bytes, "PowerComp[%04x]\n", rf_param->RSSIPowerCompOffset);
    bytes += scnprintf(&buf[bytes], size-bytes, "channel[%04x]\n", rf_param->channel);
    bytes += scnprintf(&buf[bytes], size-bytes, "power[%04x]\n", rf_param->power);
    bytes += scnprintf(&buf[bytes], size-bytes, "retry[%04x]\n", rf_param->retry);
    bytes += scnprintf(&buf[bytes], size-bytes, "gw_UID[");
    for(i=0; i < UID_MAX_SIZE; ++i) {
        bytes += scnprintf(&buf[bytes], size-bytes, "%02x", rf_param->gatewayUID[i]);
    }
    bytes += scnprintf(&buf[bytes], size-bytes, "]\n");
    bytes += scnprintf(&buf[bytes], size-bytes, "gw_panId[%04x]\n", rf_param->gatewayPanID);
    bytes += scnprintf(&buf[bytes], size-bytes, "RFID[%04x]\n", rf_param->RFID);
    bytes += scnprintf(&buf[bytes], size-bytes, "powerhilo[%04x]\n", rf_param->powerhilo);
    bytes += scnprintf(&buf[bytes], size-bytes, "lqi[%04x]\n", rf_param->LinkLQIThreshold);
    bytes += scnprintf(&buf[bytes], size-bytes, "powerBoost[%04x]\n", rf_param->powerBoost);

    bytes += scnprintf(&buf[bytes], size-bytes, " data_pipe len = %u\n", kfifo_len(&rf->data_pipe.fifo));
    bytes += scnprintf(&buf[bytes], size-bytes, "  ctl_pipe len = %u\n", kfifo_len(&rf->ctl_pipe.fifo));
    bytes += scnprintf(&buf[bytes], size-bytes, "debug_pipe len = %u\n", kfifo_len(&rf->diag_pipe.fifo));
    return bytes;
}

static ssize_t debug_register_dump(struct device *dev, struct device_attribute *attr, char *buf)
{
    u16 data[MC1319xDRV_MAX_REGS];
    int i;
    u8 addr = 0;
    for(i=0; i < MC1319xDRV_MAX_REGS; ++i, ++addr) {
        // Now read and fill the buffer
        data[i] = (UINT32) SPIDrvRead(addr);
    }
    print_hex_dump(KERN_INFO, "regs: ", DUMP_PREFIX_OFFSET,
            16, 2, data, ARRAY_SIZE(data)*sizeof(*data), false);
    return 0;
}

static ssize_t mc13192_send_test_packet(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    int bytes = ARRAY_SIZE(broadcastData);
    memcpy(testPacket.RfData.header.dstAddr, bogus_broadcastUid, 8);
    memcpy(testPacket.RfData.data, broadcastData, bytes);
    memcpy(testPacket.RfData.header.srcAddr, bogus_uid, UID_MAX_SIZE);
    bytes += sizeof(testPacket.RfData.header);
    bytes = OsQueueBytes(TxRxTask.hHandleToRFPipe, &testPacket, bytes);
    if(bytes <= 0) {
        dev_err(dev, "Failed to enqueue test packet --- err = %x!\n", bytes);
    }
    else {
        dev_err(dev, "Queued %d bytes\n", bytes);
    }
    OsSetEvent(TxRxTask.events, RF_TRX_TX_PACKET);
    return bytes;
}
DM_RESOURCE_HANDLE g_queue = 0;
static ssize_t dmqueue_create(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
//    pr_info("dmqueue_create g_queue = %u\n", g_queue);
    if(g_queue == 0) {
//        pr_info("creating queue\n");
        g_queue = OsQueueCreate(1024, 8);
    }
    return size;
}

static ssize_t dmqueue_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    INT32 bytes = OsQueueBytes(g_queue, buf, size);
    if(bytes <= 0) {
        pr_info("Failed to queue data - err = %li\n", bytes);
    }
    return (bytes > 0) ? bytes : 0;
}

static ssize_t dmqueue_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    INT32 bytes = OsDequeueBytes(g_queue, buf, PAGE_SIZE);
    if(bytes <= 0) {
        pr_info("Failed to queue data - err = %li\n", bytes);
    }
    return (bytes > 0) ? bytes : 0;
}

static ssize_t dmqueue_len(struct device *dev, struct device_attribute *attr, char *buf)
{
    UINT32 bytes = OsGetQueueCount(g_queue);
    ssize_t retval = scnprintf(buf, PAGE_SIZE, "%lu\n", bytes);
    return retval;
}

#define S_IRWUGO (S_IRUGO|S_IWUGO)
static DEVICE_ATTR(rfid,        S_IRWUGO, rfid_read, rfid_write);
static DEVICE_ATTR(uid,         S_IRWUGO, uid_read, uid_write);
static DEVICE_ATTR(panid,       S_IRWUGO, panid_read, panid_write);
static DEVICE_ATTR(gw_panid,    S_IRWUGO, gw_panid_read, gw_panid_write);
static DEVICE_ATTR(gw_uid,      S_IRWUGO, gw_uid_read, gw_uid_write);
static DEVICE_ATTR(cca_thresh,  S_IRWUGO, cca_thresh_read, cca_thresh_write);
static DEVICE_ATTR(power_comp,  S_IRWUGO, power_comp_read, power_comp_write);
static DEVICE_ATTR(channel,     S_IRWUGO, channel_read, channel_write);
static DEVICE_ATTR(retry,       S_IRWUGO, retry_read, retry_write);
static DEVICE_ATTR(power_hilo,  S_IRWUGO, power_hilo_read, power_hilo_write);
static DEVICE_ATTR(retry_count, S_IRWUGO, retry_count_read, retry_count_write);
static DEVICE_ATTR(prelink_pkt_count, S_IRWUGO, prelink_pkt_count_read, prelink_pkt_count_write);
static DEVICE_ATTR(power_boost, S_IRWUGO, power_boost_read, power_boost_write);
static DEVICE_ATTR(tsid,        S_IRWUGO, tsid_read, tsid_write);
static DEVICE_ATTR(bootid,      S_IRWUGO, bootid_read, bootid_write);

/* Actions or status */
static DEVICE_ATTR(acquire,     S_IWUGO, NULL, mc13192_acquire);
static DEVICE_ATTR(connect,     S_IWUGO, NULL, mc13192_connect);
static DEVICE_ATTR(init_done,   S_IWUGO, NULL, init_done_write);
static DEVICE_ATTR(signal_strength, S_IRUGO, signal_strength_read, NULL);
static DEVICE_ATTR(gw_connected,S_IRUGO, gw_connected_read, NULL);
static DEVICE_ATTR(cs_connected,S_IRUGO, cs_connected_read, NULL);
static DEVICE_ATTR(lqi,         S_IRWUGO, lqi_read, lqi_write);
static DEVICE_ATTR(gw_acquired, S_IRUGO, gw_acquired_read, NULL);
static DEVICE_ATTR(gw_name,     S_IRUGO, gw_name_read, NULL);


/* TODO: RF_DEBUG */
static DEVICE_ATTR(tx_packet,   S_IWUGO, NULL, mc13192_send_test_packet);
static DEVICE_ATTR(diag_desc_dump_write,S_IWUGO, NULL, diag_desc_dump_write);
static DEVICE_ATTR(debug_trace, S_IRWUGO, debug_trace_read, debug_trace_write);
static DEVICE_ATTR(show_rfparams, S_IRUGO, mc13192_show_rfparams, NULL);
static DEVICE_ATTR(register_dump, S_IRUGO, debug_register_dump, NULL);

static DEVICE_ATTR(q_create,    S_IWUGO, NULL, dmqueue_create);
static DEVICE_ATTR(q_write,     S_IWUGO, NULL, dmqueue_write);
static DEVICE_ATTR(q_read,      S_IRUGO, dmqueue_read, NULL);
static DEVICE_ATTR(q_len,       S_IRUGO, dmqueue_len, NULL);

static struct attribute *mc13192_attributes[] = {
    &dev_attr_rfid.attr,        /* Settings */
    &dev_attr_uid.attr,
    &dev_attr_panid.attr,
    &dev_attr_gw_panid.attr,
    &dev_attr_gw_uid.attr,
    &dev_attr_gw_name.attr,
    &dev_attr_cca_thresh.attr,
    &dev_attr_power_comp.attr,
    &dev_attr_channel.attr,
    &dev_attr_retry.attr,
    &dev_attr_power_hilo.attr,
    &dev_attr_retry_count.attr,
    &dev_attr_prelink_pkt_count.attr,
    &dev_attr_power_boost.attr,
    &dev_attr_tsid.attr,
    &dev_attr_bootid.attr,

    &dev_attr_acquire.attr,     /* Actions or status*/
    &dev_attr_connect.attr,
    &dev_attr_init_done.attr,
    &dev_attr_signal_strength.attr,
    &dev_attr_lqi.attr,
    &dev_attr_gw_connected.attr,
    &dev_attr_cs_connected.attr,
    &dev_attr_gw_acquired.attr,

    &dev_attr_tx_packet.attr,
    &dev_attr_diag_desc_dump_write.attr,
    &dev_attr_debug_trace.attr,
    &dev_attr_show_rfparams.attr,
    &dev_attr_register_dump.attr,
#if 0
    &dev_attr_q_create.attr, /* Debug sysfs files */
    &dev_attr_q_write.attr,
    &dev_attr_q_read.attr,
    &dev_attr_q_len.attr,
#endif
    NULL
};

static ssize_t lowlevel_resets(struct device *dev, struct device_attribute *attr, char *buf)
{
    return scnprintf(buf, PAGE_SIZE, "%lu\n", Rfdiags_getLowLevelResets());
}

static ssize_t lowlevel_rx_count(struct device *dev, struct device_attribute *attr, char *buf)
{
    return scnprintf(buf, PAGE_SIZE, "%lu\n", Rfdiags_getLowLevelRecvCnt());
}

static ssize_t lowlevel_invalid_rx_count(struct device *dev, struct device_attribute *attr, char *buf)
{
    return scnprintf(buf, PAGE_SIZE, "%lu\n", Rfdiags_getLowLevelInvalidRxPacketCnt());
}

static ssize_t lowlevel_retry_count(struct device *dev, struct device_attribute *attr, char *buf)
{
    return scnprintf(buf, PAGE_SIZE, "%lu\n", Rfdiags_getLowLevelRetryPacketCnt());
}

static ssize_t lowlevel_rxpacket_timestamp(struct device *dev, struct device_attribute *attr, char *buf)
{
    return scnprintf(buf, PAGE_SIZE, "%lu\n", g_u32PacketRxTime);
}

static ssize_t lowlevel_reset_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    SMAC_SM_ResetPhy_And_Nack();
    return size;
}

static DEVICE_ATTR(resets,              S_IRUGO, lowlevel_resets,           NULL);
static DEVICE_ATTR(rx_count,            S_IRUGO, lowlevel_rx_count,         NULL);
static DEVICE_ATTR(invalid_rx_count,    S_IRUGO, lowlevel_invalid_rx_count, NULL);
static DEVICE_ATTR(ll_retry_count,      S_IRUGO, lowlevel_retry_count,      NULL);
static DEVICE_ATTR(rxpacket_timestamp,  S_IRUGO, lowlevel_rxpacket_timestamp,      NULL);
static DEVICE_ATTR(reset,      S_IWUGO, NULL,                      lowlevel_reset_write);

static struct attribute *mc13192_lowlevel_attributes[] = {
    &dev_attr_resets.attr,
    &dev_attr_rx_count.attr,
    &dev_attr_invalid_rx_count.attr,
    &dev_attr_ll_retry_count.attr,
    &dev_attr_rxpacket_timestamp.attr,
    &dev_attr_reset.attr,
    NULL
};

static struct attribute_group mc13192_attribute_group = {
    .attrs = mc13192_attributes
};

int diag_show_info(void __user *argp)
{
    RF_DIAG_SHOW_DIAG_INFO *pShowDiagInfo;
    UINT8 IndexArray[MAX_DESCRIPTIVE_TABLE_ENTRIES];
    TYPE_RF_DESCRIPTIVE_TABLE *pDesct = 0;
    UINT16 count;
    UINT8 *pindex;

    if(!access_ok(VERIFY_WRITE, argp, sizeof(RF_DIAG_SHOW_DIAG_INFO))) {
        RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid output buffer\r\n")));
        return -EFAULT;
    }
    pShowDiagInfo = (RF_DIAG_SHOW_DIAG_INFO*)argp;

    count = RFDesc_GetIndexToDescriptiveTableAllDevices (IndexArray);
    pindex= IndexArray;
    while (count-- && (*pindex < g_MaxSupportedRFDevices))
    {
        pDesct = RFDesc_GetDescriptiveEntryIDInfo(*pindex);
        ++pindex;
    }
    if(!pDesct) {
        RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid descriptor\r\n")));
        return -EINVAL;
    }

    pShowDiagInfo->linkstatecnt = pDesct->linkstatecnt;
    pShowDiagInfo->linkqualityAtLocalDevice = pDesct->LQ.linkqualityAtLocalDevice;
    pShowDiagInfo->txpacketcount = pDesct->COMLog.txpacketcount;
    pShowDiagInfo->txpacketretrycount = pDesct->COMLog.txpacketretrycount;
    pShowDiagInfo->rxpacketcount = pDesct->COMLog.rxpacketcount;
    pShowDiagInfo->rxpacketrepeatcount = pDesct->COMLog.rxpacketrepeatcount;
    pShowDiagInfo->rxpacketsequenceErrcount = pDesct->COMLog.rxpacketsequenceErrcount;
    pShowDiagInfo->rxpacketsegmentationerrcnt = pDesct->COMLog.rxpacketsegmentationerrcnt;

    // Low level diagnostics (as that in TPS-6X setup screen diag display, sub page)
    pShowDiagInfo->rfdiag.Rfdiags_numTimer1Ints = Rfdiags_getNumTimer1Ints();
    pShowDiagInfo->rfdiag.Rfdiags_numLOUnLockInts = Rfdiags_getNumLOUnLockInts();
    pShowDiagInfo->rfdiag.Rfdiags_numInvalidPhyPacketsRecvd = Rfdiags_getNumInvalidPhyPacketsRecvd();
    pShowDiagInfo->rfdiag.Rfdiags_NumCCAFailures = Rfdiags_getNumCCAFailures();
    pShowDiagInfo->rfdiag.Rfdiags_LowLevelRecvCnt = Rfdiags_getLowLevelRecvCnt();
    pShowDiagInfo->rfdiag.Rfdiags_LowLevelRecvByteCnt = Rfdiags_getLowLevelRecvByteCnt();
    pShowDiagInfo->rfdiag.Rfdiags_LowLevelTotalLostPackets = Rfdiags_getLowLevelTotalLostPackets();
    pShowDiagInfo->rfdiag.Rfdiags_LowLevelInvalidRxPacketCnt = Rfdiags_getLowLevelInvalidRxPacketCnt();
    pShowDiagInfo->rfdiag.Rfdiags_LowLevelRetryPacketCnt = Rfdiags_getLowLevelRetryPacketCnt();
    pShowDiagInfo->rfdiag.Rfdiags_LowLevelTxPacketCnt = Rfdiags_getLowLevelTxPacketCnt();
    pShowDiagInfo->rfdiag.Rfdiags_LowLevelTxPacketBytes = Rfdiags_getLowLevelTxPacketBytes();

    // Following are not present in TPS-6X setup screen display
    pShowDiagInfo->rfdiag.Rfdiags_numCCA_TxDone = Rfdiags_getNumCCA_TxDone();
    pShowDiagInfo->rfdiag.Rfdiags_LowLevelResets = Rfdiags_getLowLevelResets();
    pShowDiagInfo->rfdiag.Rfdiags_numBadTimerISR = Rfdiags_getNumBadTimerISR();
    pShowDiagInfo->rfdiag.Rfdiags_ChannelBusyResets = Rfdiags_getChannelBusyResets();
    pShowDiagInfo->rfdiag.Rfdiags_NumDataConfMessages = Rfdiags_getNumDataConfMessages();
    pShowDiagInfo->rfdiag.Rfdiags_NumResetAndNacks = Rfdiags_getNumResetAndNacks();
    pShowDiagInfo->rfdiag.Rfdiags_Congestion = Rfdiags_getCongestion();
    pShowDiagInfo->rfdiag.Rfdiags_Utilization = Rfdiags_getUtilization();
    pShowDiagInfo->rfdiag.Rfdiags_NumCCATimeoutsFailures = Rfdiags_getNumCCATimeoutsFailures();
    pShowDiagInfo->rfdiag.Rfdiags_NumRxInactivityResets = Rfdiags_getNumRxInactivityResets();

    // PHY Information
    pShowDiagInfo->PhyInfo.PhyInitFailed = Rfdiags_getLowLevelPhyInitFailures();
    pShowDiagInfo->PhyInfo.PhyChipId = Rfdiags_getLowLevelPhyChipId();

    // On success, return size of the structure
    return 0;
}

int diag_phy_info(void __user *argp)
{
    RF_PHY_INFO* pPhyInfo;
    if(!access_ok(VERIFY_WRITE, argp, sizeof(RF_PHY_INFO))) {
        RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid output buffer\r\n")));
        return -EFAULT;
    }
    pPhyInfo = (RF_PHY_INFO*)argp;

    // Fill buffer
    pPhyInfo->PhyInitFailed = Rfdiags_getLowLevelPhyInitFailures();
    pPhyInfo->PhyChipId = Rfdiags_getLowLevelPhyChipId();
    return 0;
}

int diag_descr(void __user *argp)
{
    HPR_TY_DiagData DescrData;
    RF_DIAG_DESCR *pDiagDescr;
    TYPE_RF_DESCRIPTIVE_TABLE *pDesct;
    UNSIGNED pipesize, avaliable, messages;
    INT8    buf[100];

    // Read the max descriptive table entry.
    UINT32 dwTemp = RFDesc_MaxSupportedDescriptiveEntries();
    //RETAILMSG(1, (TEXT("RF_Diag: Descriptor (Index: %d, Max supported: %d)\r\n"), pDiagDescr->index, dwTemp));

    if(!access_ok(VERIFY_WRITE, argp, sizeof(RF_DIAG_DESCR))) {
        RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid output buffer\r\n")));
        return -EFAULT;
    }
    //TODO: fix to make sure input really has enough space?
    pDiagDescr = (RF_DIAG_DESCR*)argp;

    // Only if the index is less than the max supported index
    if (pDiagDescr->index < dwTemp) {
        pDesct = RFDesc_GetDescriptiveEntryIDInfo((UINT8)pDiagDescr->index);
        if (pDesct == NULL) {
            RETAILMSG(1, (TEXT("RF_Diagnostics: RF Descriptor NULL\r\n")));
            // Type cast the output buffer
            pDiagDescr->index = dwTemp;
            return -EFAULT;
        }
        // Fill in data and control code and send to app.
        DescrData.dwDataCode = HPRF_DIAG_DATA_DESCR_COMMAND_REPLY;

        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " [RF DESCRIPTOR REPLY]\r\n");
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Descriptive entry:                     %lu\r\n", pDiagDescr->index);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " IDstring:                              %s\r\n",   pDesct->IDInfo.idstring);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " UID:                                   %s\r\n",   ConvertUIDtoString ((UINT8*)&pDesct->IDInfo.uid, buf, gAddrModeLong_c));
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Panid:                                 %04x\r\n", pDesct->IDInfo.panid);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Addrmode:                              %d\r\n", pDesct->IDInfo.addrmode);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " RFID:                                  %02x\r\n", pDesct->IDInfo.rfid);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " TSID:                                  %08lx\r\n", pDesct->IDInfo.tsid);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Power:                                 %d\r\n", pDesct->IDInfo.power);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Deviceoptions:                         %d\r\n", pDesct->IDInfo.deviceoptions);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link state:                            %s\r\n",   ConvertLinkStatetoString (pDesct->linkstate, buf));
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link state connection count:           %lu\r\n", pDesct->linkstatecnt);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        //  new
        ConvertTimeToHMS((OS_RetrieveClock() - pDesct->linkeactiveestablishedtime), buf);
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link active time:                      %lu (%s)\r\n", pDesct->linkeactiveestablishedtime, buf);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        //  new
        RFTransmit_GetPipeAvailableMemorySizeByIndex ((UINT8)pDiagDescr->index, &pipesize, &avaliable, &messages);
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Transmit Pipe info size/avaliable/msgs:%lu/%lu/%lu\r\n", pipesize,avaliable,messages);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Linkquality at local device:           %d (%d dBm)\r\n", pDesct->LQ.linkqualityAtLocalDevice, RFReceive_ConvertLinkQualityTodBm(pDesct->LQ.linkqualityAtLocalDevice));
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Linkquality at remote device:          %d (%d dBm)\r\n", pDesct->LQ.linkqualityAtRemoteDevice, RFReceive_ConvertLinkQualityTodBm(pDesct->LQ.linkqualityAtRemoteDevice));
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Waiting for ack flag:                  %d\r\n",   pDesct->waitingforACK);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code available flag:            %d\r\n",   pDesct->resultcodeavailable);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code:                           %d\r\n", pDesct->resultcode);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Retry timer:                           %u\r\n", pDesct->RetryTimer);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Retry timed out:                       %d\r\n", pDesct->RetryTimedOut);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet retry cntr:                  %d\r\n", pDesct->txpacketretrycntr);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Maximum Tx packet retries:             %d\r\n", pDesct->maxtxpacketretries);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Heart beat timer:                      %d\r\n", pDesct->HeartBeatTimer);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Heart beat timed out:                  %d\r\n",   pDesct->HeartBeatTimedOut);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code timer:                     %d\r\n", pDesct->ResultCodeTimer);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code timed out:                 %d\r\n",   pDesct->ResultCodeTimedOut);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet sequence number:             %d\r\n", pDesct->txsequence);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packets:                %lu\r\n", pDesct->COMLog.txpacketcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packet retries:         %lu\r\n", pDesct->COMLog.txpacketretrycount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packet in errors:       %lu\r\n", pDesct->COMLog.txpacketerrorcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of times tx buffer overflow:  %lu\r\n", pDesct->COMLog.txbufoverflowcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet resultcode timeouts:   %lu\r\n", pDesct->COMLog.txpacketresultcodetimeouts);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        // new
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of concatenated tx packet:    %lu\r\n", pDesct->COMLog.txpacketconcatcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of segmented tx packet:       %lu\r\n", pDesct->COMLog.txpacketsegmented);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        //printf(" tx packet                              %u\r\n", pDesct->COMLog.txpacketfailcount);
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet payload data in bytes: %lu\r\n", pDesct->COMLog.txpacketpayloadbytecount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet overhead in bytes:     %lu\r\n", pDesct->COMLog.txpacketoverheadbytecount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        //  modified
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Average tx packet length in bytes:     %lu\r\n", pDesct->COMLog.txpacketcount?
                                                    ((pDesct->COMLog.txpacketoverheadbytecount+pDesct->COMLog.txpacketpayloadbytecount)/pDesct->COMLog.txpacketcount):0);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        //  new
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet turn around delay in 10msec: %lu\r\n",    pDesct->COMLog.txpacketturnaroundtime);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet avg turn around delay in 10msec: %lu\r\n",    (pDesct->COMLog.txpacketcount-pDesct->COMLog.txpacketretrycount)?(pDesct->COMLog.txpacketaccturnaroundtime/(pDesct->COMLog.txpacketcount-pDesct->COMLog.txpacketretrycount)):0);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Rx packet sequence number:             %d\r\n",  pDesct->rxsequence);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packets:                %lu\r\n",  pDesct->COMLog.rxpacketcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet repeats:         %lu\r\n",  pDesct->COMLog.rxpacketrepeatcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet sequence err:    %lu\r\n",  pDesct->COMLog.rxpacketsequenceErrcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet segment err:     %lu\r\n",  pDesct->COMLog.rxpacketsegmentationerrcnt);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet with invalid len:%lu\r\n",  pDesct->COMLog.rxpacketlenerror);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet with hdr/footr err:%lu\r\n",    pDesct->COMLog.rxpackethdrftrerror);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of times gw rx buffer overflow: %lu\r\n",  pDesct->COMLog.rxbufoverflowcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        // new
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet concatenation:   %lu\r\n", pDesct->COMLog.rxpacketconcatcount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx acks:                   %lu\r\n", pDesct->COMLog.rxpacketacks);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx naks/cca's:             %lu\r\n", pDesct->COMLog.rxpacketnaks);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total rx packet payload data in bytes: %lu\r\n",    pDesct->COMLog.rxpacketpayloadbytecount);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total rx gw startup packets:           %lu\r\n",    pDesct->COMLog.rxgwrestarts);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of heartbeat timeouts:        %lu\r\n",    pDesct->COMLog.heartbeattimeouts);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no tx cong. avoidance:           %lu\r\n", pDesct->COMLog.txpacketcongavoidance);
        SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

        return 0;
    }
    else {
        RETAILMSG(1, (TEXT("RF_Diagnostics: RF Descriptor out of bound\r\n")));
        pDiagDescr->index = dwTemp;
        return FALSE;
    }
}

int diag_rf_stack_trace(void __user *argp)
{
    UINT32 dwTemp = 0;
    TYPE_RF_STACK_TRACE* pStackTrace;
    RETAILMSG(1, (TEXT("RF_Diag: Stack trace.\r\n")));

    if(!access_ok(VERIFY_WRITE, argp, sizeof(TYPE_RF_STACK_TRACE))) {
        RETAILMSG(1, (TEXT("RF_Diag: Invalid stack trace buffer.\r\n")));
        return -EFAULT;
    }
    pStackTrace = (TYPE_RF_STACK_TRACE*)argp;

    pStackTrace->SMAC_SM_Debug_State_Ctr = SMAC_SM_Debug_State_Ctr;
    for (dwTemp = 0; dwTemp < SMAC_SM_DEBUG_NUMENTRIES; dwTemp++)
    {
        pStackTrace->SMAC_SM_Debug_State[dwTemp] = SMAC_SM_Debug_State[dwTemp];
        pStackTrace->SMAC_SM_Debug_Timestamp[dwTemp] = SMAC_SM_Debug_Timestamp[dwTemp];
    }
    return 0;
}

int diag_rf_clr_diag(void __user *argp)
{
    RETAILMSG(1, (TEXT("RF_Diag: Clear RF diag counters.\r\n")));
    //  Clear diagnostic counters
    RFAPP_ClearDiagCounters();
    return 0;
}

int diag_rf_stack_diag(void __user *argp)
{
    TYPE_RF_STACK_DIAG* pStackDiag;
    UINT32 dwTemp = 0;
    UINT8 msgindex, msgentries, errpktcnt;

    RETAILMSG(1, (TEXT("RF_Diag: Stack diag.\r\n")));
    if(!access_ok(VERIFY_WRITE, argp, sizeof(TYPE_RF_STACK_DIAG))) {
        RETAILMSG(1, (TEXT("RF_Diag: Invalid stack diag buffer\r\n")));
        return -EFAULT;
    }
    pStackDiag = (TYPE_RF_STACK_DIAG*)argp;

    // Fill the data
    pStackDiag->rfdiag.Rfdiags_numTimer1Ints = Rfdiags_getNumTimer1Ints();
    pStackDiag->rfdiag.Rfdiags_numLOUnLockInts = Rfdiags_getNumLOUnLockInts();
    pStackDiag->rfdiag.Rfdiags_numInvalidPhyPacketsRecvd = Rfdiags_getNumInvalidPhyPacketsRecvd();
    pStackDiag->rfdiag.Rfdiags_numCCA_TxDone = Rfdiags_getNumCCA_TxDone();
    pStackDiag->rfdiag.Rfdiags_LowLevelRecvCnt = Rfdiags_getLowLevelRecvCnt();
    pStackDiag->rfdiag.Rfdiags_LowLevelRecvByteCnt = Rfdiags_getLowLevelRecvByteCnt();
    pStackDiag->rfdiag.Rfdiags_LowLevelTotalLostPackets = Rfdiags_getLowLevelTotalLostPackets();
    pStackDiag->rfdiag.Rfdiags_LowLevelInvalidRxPacketCnt = Rfdiags_getLowLevelInvalidRxPacketCnt();
    pStackDiag->rfdiag.Rfdiags_LowLevelRetryPacketCnt = Rfdiags_getLowLevelRetryPacketCnt();
    pStackDiag->rfdiag.Rfdiags_LowLevelTxPacketCnt = Rfdiags_getLowLevelTxPacketCnt();
    pStackDiag->rfdiag.Rfdiags_LowLevelTxPacketBytes = Rfdiags_getLowLevelTxPacketBytes();
    pStackDiag->rfdiag.Rfdiags_LowLevelResets = Rfdiags_getLowLevelResets();
    pStackDiag->rfdiag.Rfdiags_numBadTimerISR = Rfdiags_getNumBadTimerISR();
    pStackDiag->rfdiag.Rfdiags_ChannelBusyResets = Rfdiags_getChannelBusyResets();
    pStackDiag->rfdiag.Rfdiags_NumDataConfMessages = Rfdiags_getNumDataConfMessages();
    pStackDiag->rfdiag.Rfdiags_NumResetAndNacks = Rfdiags_getNumResetAndNacks();
    pStackDiag->rfdiag.Rfdiags_Congestion = Rfdiags_getCongestion();
    pStackDiag->rfdiag.Rfdiags_Utilization = Rfdiags_getUtilization();
    pStackDiag->rfdiag.Rfdiags_NumCCAFailures = Rfdiags_getNumCCAFailures();
    pStackDiag->rfdiag.Rfdiags_NumCCATimeoutsFailures = Rfdiags_getNumCCATimeoutsFailures();
    pStackDiag->rfdiag.Rfdiags_NumRxInactivityResets = Rfdiags_getNumRxInactivityResets();

    for (dwTemp = 0; dwTemp < g_MaxSupportedRFDevices; dwTemp++)
        pStackDiag->numResCodeTimeouts += g_pRFDeviceDescriptiveTable[dwTemp].COMLog.txpacketresultcodetimeouts;

    RFDesc_GetErrorPktIndex (DEVICE_DESCRIPTIVE_TABLE_INDEX, &msgindex, &msgentries, &errpktcnt);
    pStackDiag->errpktcnt = (UINT32)errpktcnt;
    return 0;
}

int diag_rf_phy_read(void __user *argp)
{
    //if (!pBufOut || dwLenIn < sizeof (TYPE_RF_PHY_READ)|| dwLenOut < sizeof (TYPE_RF_PHY_READ))
    UINT32 dwTemp = 0;
    UINT8 addr;
    UINT16 count;
    TYPE_RF_PHY_READ phyReadIn;
    TYPE_RF_PHY_READ *pPhyReadOut;
    if(copy_from_user(&phyReadIn, argp, sizeof(TYPE_RF_PHY_READ)) ||
       !access_ok(VERIFY_WRITE, argp, sizeof(TYPE_RF_PHY_READ))) {
        RETAILMSG(1, (TEXT("RF_Diag: Invalid RF PHY In-Out buffers.\r\n")));
        return -EFAULT;
    }
    // Type cast in and out buffers (should be of the same type.
//    pPhyReadIn = (TYPE_RF_PHY_READ*)pBufIn;
    pPhyReadOut = (TYPE_RF_PHY_READ*)argp;

    // Start at the Read address....
    dwTemp = 0;
//    addr = (UINT8)pPhyReadIn->ReadAddr;
//    count = (UINT16)pPhyReadIn->ReadCount;
    addr = (UINT8)0;
    count = (UINT16)MC1319xDRV_MAX_REGS;
    //RETAILMSG(1, (TEXT("RF_Diag: Read PHY (Reg: 0x%02x, Count: %d)\r\n"),addr, count));

    while (count < MC1319xDRV_MAX_REGS)
    {
        // Now read and fill the buffer
        pPhyReadOut->Data[dwTemp] = (UINT32) SPIDrvRead(addr);
        count--;
        addr++;
        dwTemp++;
    }
    return (sizeof (TYPE_RF_PHY_READ));
}

int diag_rf_netlink(void __user *argp)
{
    TYPE_RF_NETLINK_INFO* pRFNetlinkInfo;
    UINT8 IndexArray[MAX_DESCRIPTIVE_TABLE_ENTRIES];
    TYPE_RF_DESCRIPTIVE_TABLE *pDesct;
    UINT8 *pindex;
    UINT16 count;
    RETAILMSG(1, (TEXT("RF_Diag: RF Netlink.\r\n")));

    if(!access_ok(VERIFY_WRITE, argp, sizeof(TYPE_RF_NETLINK_INFO))) {
        RETAILMSG(1, (TEXT("RF_Diag: Invalid RF NETLINK Out buffer.\r\n")));
        return -EFAULT;
    }
    pRFNetlinkInfo = (TYPE_RF_NETLINK_INFO*)argp;

    // Fill the info..
    pRFNetlinkInfo->GwConn = RFIntRFApp.pF_GatewayConnectionStatusQuery();
    pRFNetlinkInfo->CtrlSysConn = RFIntRFApp.pF_ControlSystemConnectionStatusQuery();
    if (RFReceive_IsControlSystemConnectedNetTypeCresnet()) {
        pRFNetlinkInfo->GwCtrlConnType = 1;
    }
    else if (RFReceive_IsControlSystemConnectedNetTypeEthernet()) {
        pRFNetlinkInfo->GwCtrlConnType = 2;
    }
    else {
        pRFNetlinkInfo->GwCtrlConnType = 0;
    }

    IndexArray[0] = DEVICE_DESCRIPTIVE_TABLE_INDEX;
    pindex = IndexArray;
    count = 1;

    RFDesc_SortIndexElementsByRFID (pindex, (UINT8)count);
    pDesct = RFDesc_GetDescriptiveEntryIDInfo(*pindex);
    if (pDesct)
    {
        ConvertLinkStatetoString (g_pRFDeviceDescriptiveTable[*pindex].linkstate, pRFNetlinkInfo->LinkStateStr);
        pRFNetlinkInfo->TimeStr[0] = 0;
        if (pDesct->linkstate == RF_DEVICE_LINK_STATE_ACTIVE)
        {
            ConvertTimeToHMS((OS_RetrieveClock() - pDesct->linkeactiveestablishedtime), pRFNetlinkInfo->TimeStr);
        }
        pRFNetlinkInfo->GwCnetId = g_pRFDeviceDescriptiveTable[*pindex].IDInfo.rfid;
        pRFNetlinkInfo->LinkCount = pDesct->linkstatecnt;
        pRFNetlinkInfo->RxLQIinDev = RFReceive_ConvertLinkQualityToPercent(pDesct->LQ.linkqualityAtLocalDevice, LQI_DISP_OFFSET);
        pRFNetlinkInfo->RxLQIinGW = RFReceive_ConvertLinkQualityToPercent(pDesct->LQ.linkqualityAtRemoteDevice, LQI_DISP_OFFSET);
    }
    return 0;
}

int diag_rf_version(void __user *argp)
{
    TYPE_RF_VERSION_INFO* pRfVersion;
    RETAILMSG(1, (TEXT("RF_Diag: RF Version.\r\n")));

    if(!access_ok(VERIFY_WRITE, argp, sizeof(TYPE_RF_VERSION_INFO))) {
        RETAILMSG(1, (TEXT("RF_Diag: Invalid RF Version Out buffer.\r\n")));
        return -EFAULT;
    }
    pRfVersion = (TYPE_RF_VERSION_INFO*)argp;

    strcpy(pRfVersion->RFVersion, RF_Version);
    strcpy(pRfVersion->SmacVersion, bogus_smac_version);
    return 0;
}

int diag_rf_power_dbm(void __user *argp)
{
    TYPE_RF_POWER_DBM rfPowerDbmIn;
    TYPE_RF_POWER_DBM* pRfPowerDbmOut;
    //TODO: FIXME: Float variables
//    FLOAT fTemp = 0;
    UINT32 fTemp = 0;
    if(copy_from_user(&rfPowerDbmIn, argp, sizeof(TYPE_RF_POWER_DBM)) ||
       !access_ok(VERIFY_WRITE, argp, sizeof(TYPE_RF_POWER_DBM))) {
        RETAILMSG(1, (TEXT("RF_Diag: Invalid RF Power level dBm in or out buffer.\r\n")));
        return -EFAULT;
    }
    pRfPowerDbmOut = (TYPE_RF_POWER_DBM*)argp;

//        fTemp = RFIntRFApp.pF_ConvertPowerTodBm((UINT16)pRfPowerDbm->Power, (BOOL)pRfPowerDbm->Translate);
    fTemp = rfPowerDbmIn.Power;

    // Send the output back
    pRfPowerDbmOut->Power_dBm = fTemp;

    RETAILMSG(1, (TEXT("RF_Diag: RF Power dBm: %2.2f\r\n"), pRfPowerDbmOut->Power_dBm));
    return 0;
}

int rf_err_log(void __user *argp)
{
#ifdef HPRF_ERR_MSG_LOG
    UINT32 dwTemp = 0;
    HPR_TY_DiagData Diag;
    TYPE_RF_DIAG_ERR_LOG* pErrLog;
    char *pStrPtr;
    RETAILMSG(1, (TEXT("RF_Diag: RF Error Log (Msgs: %u, Total bytes: %u.\r\n"),
                RfErrorLog.NoErrorMsgs, RfErrorLog.TotalBytes));
    if(!access_ok(VERIFY_WRITE, argp, sizeof(TYPE_RF_DIAG_ERR_LOG))) {
        RETAILMSG(1, (TEXT("RF_Diag: Incorrect ERRORLOG input buffer.\r\n")));
        return -EFAULT;
    }
    pErrLog = (TYPE_RF_DIAG_ERR_LOG*)argp;

    // Fill the control code...
    Diag.dwDataCode = HPRF_DIAG_DATA_CONSOLE_PRINT;
    // If we have atleast one error messages logged.....
    if (RfErrorLog.NoErrorMsgs)
    {
        // Start from beginning...
        pStrPtr = RfErrorLog.RFErrorMsgBuff;
        // Send all messages to application...
        for (dwTemp = 0; dwTemp < RfErrorLog.NoErrorMsgs; dwTemp++)
        {
            strcpy(Diag.Data.Console.buffer, pStrPtr);
            Diag.Data.Console.dwStrLen = strlen(pStrPtr);

            // Send the string to application...
            SendDiagDataToApp((void*)&Diag, sizeof (HPR_TY_DiagData));

            // Advance the pointer to next string... (consider NULL)
            pStrPtr = pStrPtr + Diag.Data.Console.dwStrLen + 1;
        }
        // clear command...
        if (pErrLog->Control == 1)
        {
            // clear the error log...
            RfErrorLog.NoErrorMsgs = 0;
            RfErrorLog.TotalBytes = 0;
            // Send clear message...
            strcpy(Diag.Data.Console.buffer, "\r\nRF Error logs cleared!");
            Diag.Data.Console.dwStrLen = strlen(Diag.Data.Console.buffer);
            SendDiagDataToApp((void*)&Diag, sizeof (HPR_TY_DiagData));
        }
    }
    else
    {
        strcpy(Diag.Data.Console.buffer, "\r\nNo RF Error Messages logged!");
        Diag.Data.Console.dwStrLen = strlen(Diag.Data.Console.buffer);
        SendDiagDataToApp((void*)&Diag, sizeof (HPR_TY_DiagData));
    }
#endif
    return 0;
}

/**
 * Handles all the RF diagnostic functions.
 * pBufIn Input buffer
 * dwLenIn Input buffer length
 * pBufOut Output buffer to return data
 * dwLenOut Length of the output buffer
 * Returns true on success, false on failure.
 */
//UINT32 RF_Diagnostics(void* pBufIn, UINT32 dwLenIn, void* pBufOut, UINT32 dwLenOut)
int RF_Diagnostics(void __user *argp)
{
    RF_DIAG_CTRL diagCtrl;

    if(copy_from_user(&diagCtrl, argp, sizeof(diagCtrl)) != 0) {
        // Invalid buffer
        RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid input buffer\r\n")));
        return -EFAULT;
    }

    switch(diagCtrl.dwCode) {
    case HPR_DIAG_CTRL_SHOW_DIAG_INFO: return diag_show_info(argp);
    case HPR_DIAG_CTRL_PHY_INFO: return diag_phy_info(argp);
    case HPR_DIAG_CTRL_DESCR: return diag_descr(argp);
    case HPR_DIAG_CTRL_RF_STACK_TRACE: return diag_rf_stack_trace(argp);
    case HPR_DIAG_CTRL_RF_CLR_DIAG: return diag_rf_clr_diag(argp);
    case HPR_DIAG_CTRL_RF_STACK_DIAG: return diag_rf_stack_diag(argp);
    case HPR_DIAG_CTRL_RF_PHY_READ: return diag_rf_phy_read(argp);
#ifdef HPRF_ERR_MSG_LOG
    case HPR_DIAG_CTRL_RF_ERR_LOG: return rf_err_log(argp);
#endif
    case HPR_DIAG_CTRL_RF_NETLINK: return diag_rf_netlink(argp);
    case HPR_DIAG_CTRL_RF_VERSION: return diag_rf_version(argp);
    case HPR_DIAG_CTRL_RF_POWER_DBM: return diag_rf_power_dbm(argp);
    default: break;
    }
    // Unknown, if it reaches here.
    RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid control code: %u\r\n"), diagCtrl.dwCode));
    return -EINVAL;
}

static int mc13192_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    int result = 0; // ASSUME success.
    UINT32 timeout;
    UINT8 connStatus;
    UINT8 uid[UID_MAX_SIZE] = {}; // null uid
    UINT16 lq_norm;
    UINT8 debugFlags = 0;
    struct ioctl_idString idString;

#ifdef MC_USE_CDEV
    struct cdev *c = filp->private_data;
#else
    struct miscdevice *c = filp->private_data;
#endif
    struct mc13192 *rf = container_of(c, struct mc13192, data_pipe.cdev);
    RF_PARAM *rfparam = &rf->rfparam;
    RF_PARAM_LOCAL *rfparamlocal = &rf->rfparamlocal;

    switch(cmd) {
    case HPR_IOCTL_ACQUIRE_START:
        if(copy_from_user(&timeout, (void __user*)arg, sizeof(timeout)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_AcquireStart(timeout);
        }
        break;
    case HPR_IOCTL_ACQUIRE_STOP:
        RFIntRFApp.pF_AcquireStop(1);
        break;
    case HPR_IOCTL_GET_RF_ID:
        if(copy_to_user((void __user*)arg, &rfparam->RFID, sizeof(rfparam->RFID)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_SET_RF_ID:
        if(copy_from_user(&rfparam->RFID, (void __user*)arg, sizeof(rfparam->RFID)) != 0) {
            pr_err("Failed to copy RFID\n");
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_GET_RF_CHANNEL:
        if(copy_to_user((void __user*)arg, &rfparam->channel, sizeof(rfparam->channel)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_SET_RF_CHANNEL:
        if(copy_from_user(&rfparam->channel, (void __user*)arg, sizeof(rfparam->channel)) != 0) {
            pr_err("Failed to copy RF channel\n");
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_GET_RF_POWER_HI_LOW:
        if(copy_to_user((void __user*)arg, &rfparam->powerhilo, sizeof(rfparam->powerhilo)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_SET_RF_POWER_HI_LOW:
        if(copy_from_user(&rfparam->powerhilo, (void __user*)arg, sizeof(rfparam->powerhilo)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetPowerLevel(rfparam->powerhilo);
        }
        break;
    case HPR_IOCTL_RF_APP_INIT_DONE:
        rf->isAppInitDone = TRUE;
        if (rfparamlocal->UID_Valid == UID_SIGN)
        {
            //RFAPP_LinkRequestHandlerStop();
            RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();
        }
        break;
    case HPR_IOCTL_GET_GW_UID:
        if(copy_to_user((void __user*)arg, &rfparam->gatewayUID, sizeof(rfparam->gatewayUID)) != 0) {
            result = -EFAULT;
        }
        break;
//HPR_IOCTL_GET_GW_UID_STRING:
    case HPR_IOCTL_SET_GW_UID:
        if(copy_from_user(&rfparam->gatewayUID, (void __user*)arg, sizeof(rfparam->gatewayUID)) != 0) {
            result = -EFAULT;
        }
        else {
            // Inform the driver (needed when the restore is done from the INI file after a OS update)
            RFIntRFApp.pF_SetGatewayUID_PANID_ADDRMODE(TRUE, &rfparam->gatewayUID[0], rfparam->gatewayPanID, rfparam->addrMode);
        }
        break;
    case HPR_IOCTL_GET_UID:
        if(copy_to_user((void __user*)arg, &rfparam->gatewayUID, sizeof(rfparam->gatewayUID)) != 0) {
            result = -EFAULT;
        }
        break;
//HPR_IOCTL_GET_UID_STRING:
    case HPR_IOCTL_SET_UID:
        if(copy_from_user(&rfparam->UID, (void __user*)arg, sizeof(rfparam->UID)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetLocaldeviceUID_PANID(TRUE, &rfparam->UID[0], rfparam->panID);
            rfparamlocal->UID_Valid = UID_SIGN;
        }
        break;
    case HPR_IOCTL_GW_CONN_QUERY:
        connStatus = RFIntRFApp.pF_GatewayConnectionStatusQuery();
        if(copy_to_user((void __user*)arg, &connStatus, sizeof(connStatus)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_CS_CONN_QUERY:
        connStatus = RFIntRFApp.pF_ControlSystemConnectionStatusQuery();
        if(copy_to_user((void __user*)arg, &connStatus, sizeof(connStatus)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_GW_ACQUIRED_QUERY:
        // 1 is acquired, 0 is not
        connStatus = !memcmp(rfparam->gatewayUID, uid, sizeof(uid));
        if(copy_to_user((void __user*)arg, &connStatus, sizeof(connStatus)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_GET_GW_NAME:
        if(copy_to_user((void __user*)arg, &rfparamlocal->gatewayName, sizeof(rfparamlocal->gatewayName)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_GET_SIGNAL_STRENGTH:
        if (!rf->lqiActive) {
            RFIntRFApp.pF_GetLQI (0, TRUE, &connStatus);
            lq_norm= RFIntRFApp.pF_ConvertLQIToIEEEScaleWord(connStatus, LQI_DISP_OFFSET);
        }
        else {
            lq_norm = 0;
        }
        if (copy_to_user((void __user*) arg, &lq_norm, sizeof(lq_norm)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_LQI_START:
        rf->localDevLqi = 255;
        rf->remoteLqi = 255;
        rf->lqiActive = TRUE;
        //TODO: Implement HPRDevice
        //OsSignalEvent(pHPRDevice->hLQIEvent);
        break;
    case HPR_IOCTL_LQI_STOP:
        rf->lqiActive = FALSE;
        break;
    case HPR_IOCTL_GET_RF_PWR_BOOST:
        if(copy_to_user((void __user*)arg, &rfparam->powerBoost, sizeof(rfparam->powerBoost)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_SET_RF_PWR_BOOST:
        if(copy_from_user(&rfparam->powerBoost, (void __user*)arg, sizeof(rfparam->powerBoost)) != 0) {
            result = -EFAULT;
        }
        else {
            //TODO: I don't understand this logic
            // min() doesn't work with an rvalue.
            rfparam->powerBoost = (rfparam->powerBoost < 21) ? rfparam->powerBoost : 21;
            // Update the main variable.
            gPowerBoost = rfparam->powerBoost;
            // Just like TPS-6X
            if (RFIntRFApp.pF_RFReceive_GetLocalDevicePower() >= RF_POWER_HI) {
                RFIntRFApp.pF_SetPowerLevel(RF_POWER_HI);
            }
            else {
                RFIntRFApp.pF_SetPowerLevel(RF_POWER_LO);
            }
        }
        break;
    case HPR_IOCTL_GET_RF_DEBUG:
        debugFlags = rf->rfparam.RFTraceFlags;
        if(copy_to_user((void __user*)arg, &debugFlags, sizeof(debugFlags)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_SET_RF_DEBUG:
        if(copy_from_user(&debugFlags, (void __user*)arg, sizeof(debugFlags)) != 0) {
            result = -EFAULT;
        }
        else {
            rf->rfparam.RFTraceFlags = debugFlags & (RF_RFTX_DEBUG_FLAG | RF_RFRX_DEBUG_FLAG | RF_GENERIC_DEBUG_FLAG);
        }
        break;
    case HPR_IOCTL_GET_PAN_ID:
        if(copy_to_user((void __user*)arg, &rfparam->panID, sizeof(rfparam->panID)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_SET_PAN_ID:
        if(copy_from_user(&rfparam->panID, (void __user*)arg, sizeof(rfparam->panID)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetLocaldeviceUID_PANID(TRUE, &rfparam->UID[0], rfparam->panID);
        }
        break;
    case HPR_GET_IOCTL_GATEWAY_PAN_ID:
        if(copy_to_user((void __user*)arg, &rfparam->gatewayPanID, sizeof(rfparam->gatewayPanID)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_SET_IOCTL_GATEWAY_PAN_ID:
        if(copy_from_user(&rfparam->gatewayPanID, (void __user*)arg, sizeof(rfparam->gatewayPanID)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetGatewayUID_PANID_ADDRMODE(TRUE, &rfparam->gatewayUID[0], rfparam->gatewayPanID, rfparam->addrMode);
        }
        break;
    case HPR_GET_IOCTL_CCA_THRESHOLD:
        if(copy_to_user((void __user*)arg, &rfparam->CCAEnergyDetectThr, sizeof(rfparam->CCAEnergyDetectThr)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_SET_IOCTL_CCA_THRESHOLD:
        if(copy_from_user(&rfparam->CCAEnergyDetectThr, (void __user*)arg, sizeof(rfparam->CCAEnergyDetectThr)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetCCAEnergyDetectThr(rfparam->CCAEnergyDetectThr);
        }
        break;
    case HPR_GET_IOCTL_RSSI_COMP_OFFSET:
        if(copy_to_user((void __user*)arg, &rfparam->RSSIPowerCompOffset, sizeof(rfparam->RSSIPowerCompOffset)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_SET_IOCTL_RSSI_COMP_OFFSET:
        if(copy_from_user(&rfparam->RSSIPowerCompOffset, (void __user*)arg, sizeof(rfparam->RSSIPowerCompOffset)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetRSSIPowerCompOffset(rfparam->RSSIPowerCompOffset);
        }
        break;
    case HPR_GET_IOCTL_RF_RETRY:
        if(copy_to_user((void __user*)arg, &rfparam->retry, sizeof(rfparam->retry)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_SET_IOCTL_RF_RETRY:
        if(copy_from_user(&rfparam->retry, (void __user*)arg, sizeof(rfparam->retry)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetRetryCount(rfparam->retry);
        }
        break;
    case HPR_IOCTL_GET_RETRY_COUNT_ON_CONN:
        if(copy_to_user((void __user*)arg, &rfparam->txPktretryonconnection, sizeof(rfparam->txPktretryonconnection)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_IOCTL_SET_RETRY_COUNT_ON_CONN:
        if(copy_from_user(&rfparam->txPktretryonconnection, (void __user*)arg, sizeof(rfparam->txPktretryonconnection)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetRetryCountOnConnectionReq(rfparam->txPktretryonconnection);
        }
        break;
    case HPR_GET_IOCTL_PRE_LINK_COUNT:
        if(copy_to_user((void __user*)arg, &rfparam->PreLinkQueryPacketCnt, sizeof(rfparam->PreLinkQueryPacketCnt)) != 0) {
            result = -EFAULT;
        }
        break;
    case HPR_SET_IOCTL_PRE_LINK_COUNT:
        if(copy_from_user(&rfparam->PreLinkQueryPacketCnt, (void __user*)arg, sizeof(rfparam->PreLinkQueryPacketCnt)) != 0) {
            result = -EFAULT;
        }
        else {
            RFIntRFApp.pF_SetPreLinkQueryPacketCnt(rfparam->PreLinkQueryPacketCnt);
        }
        break;
    case HPR_IOCTL_UPDATE_RADIO_DATA:
        if(copy_from_user(&idString, (void __user*)arg, sizeof(idString)) != 0) {
            result = -EFAULT;
        }
        else {
            rfparamlocal->TSID = idString.tsid;
            strncpy(rf->bootIdMsg, idString.bootIdMsg, MAX_BOOT_ID_MESSAGE);
        }
        break;
//    case HPR_IOCTL_RF_TESTS:
//        *pdwActualOut = RF_Tests(pBufIn, dwLenIn, pBufOut, dwLenOut);
//        break;
    case HPR_IOCTL_DIAGNOSTICS:
        pr_debug("HPRF.DLL: IOCTL - RF Diagnostics.\n");
        result = RF_Diagnostics((void __user*)arg);
        break;
    default:
        return -EINVAL;
    }

    return result;
}

static int mc13192_data_open(struct inode *inode, struct file *filp)
{
    struct mc13192 *rf = container_of(inode->i_cdev, struct mc13192, data_pipe.cdev);
    filp->private_data = rf;

//    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
//    struct mc13192 *hprf = spi_get_drvdata(spi);
//    pr_info("<----- %s -----> hprf_spi rf = %p  rf = %p\n", __FUNCTION__, hprf, rf);
    return 0;
}
static int mc13192_ctl_open(struct inode *inode, struct file *filp)
{
    struct mc13192 *rf = container_of(inode->i_cdev, struct mc13192, ctl_pipe.cdev);
    filp->private_data = rf;
    return 0;
}
static int mc13192_diag_open(struct inode *inode, struct file *filp)
{
    struct mc13192 *rf = container_of(inode->i_cdev, struct mc13192, diag_pipe.cdev);
    filp->private_data = rf;
    return 0;
}

static int mc13192_release(struct inode *inode, struct file *filp)
{
    return 0;
}

/*
 * Waits interruptibly for data to be in the pipe that is being read from.
 */
ssize_t wait_for_data(struct mc13192_pipe_dev *pipe, char __user *buf, size_t count)
{
    int err;
    size_t bytesCopied;

    if(mutex_lock_interruptible(&pipe->read_mutex)) {
        return -ERESTARTSYS;
    }
    /* Wait blocking until data arrives. */
    while(!fifo_len_spinlocked(&pipe->fifo, &pipe->slock)) {
        mutex_unlock(&pipe->read_mutex);
        if( wait_event_interruptible(pipe->wq, fifo_len_spinlocked(&pipe->fifo, &pipe->slock)) )
            return -ERESTARTSYS;

        if(mutex_lock_interruptible(&pipe->read_mutex)) {
            return -ERESTARTSYS;
        }
    }
    /* TODO: Possibly shouldn't warn on this... debugging measure. */
    if(count > HPR_MAX_PIPE_PACKET_SIZE) {
        dev_warn(&pipe->dev, "%u is greater than the maximum receive size %u.\n",
                count, HPR_MAX_PIPE_PACKET_SIZE);
        count = HPR_MAX_PIPE_PACKET_SIZE;
    }
    bytesCopied = kfifo_out_spinlocked(&pipe->fifo, pipe->tmp_read_buf, count, &pipe->slock);
    if(!bytesCopied) {
        dev_err(pipe->dev, "failed to read from kfifo.\n", bytesCopied);
    }
    err = copy_to_user(buf, pipe->tmp_read_buf, bytesCopied);
    if(err) {
        dev_err(pipe->dev, "failed to copy %u bytes to user buffer\n", err);
        bytesCopied = err;
    }
    mutex_unlock(&pipe->read_mutex);
    return bytesCopied;
}

static ssize_t mc13192_data_read(struct file *filp, char __user *buf,
                            size_t count, loff_t *ppos)
{
    struct mc13192 *rf = filp->private_data;
    return wait_for_data(&rf->data_pipe, buf, count);
}

static ssize_t mc13192_ctl_read(struct file *filp, char __user *buf,
                            size_t count, loff_t *ppos)
{
    struct mc13192 *rf = filp->private_data;
    return wait_for_data(&rf->ctl_pipe, buf, count);
}

static ssize_t mc13192_diag_read(struct file *filp, char __user *buf,
                            size_t count, loff_t *ppos)
{
    struct mc13192 *rf = filp->private_data;
    return wait_for_data(&rf->diag_pipe, buf, count);
}

static ssize_t mc13192_data_write(struct file *filp, const char __user *buf,
                             size_t count, loff_t *ppos)
{
    int err;
    struct mc13192 *rf = filp->private_data;
    char *kbuf;

    kbuf = kmalloc(count, GFP_KERNEL);
    if(!kbuf) {
        dev_err(rf->data_pipe.dev, "Failed to allocate kernel memory.\n");
        return -ENOMEM;
    }
    err = copy_from_user(kbuf, buf, count);
    if(err) {
        dev_err(rf->data_pipe.dev, "Failed to copy user buffer.\n");
        kfree(kbuf);
        return err;
    }
    mutex_lock(&rf->data_pipe.write_mutex);
    err = RFIntRFApp.pF_PostCresnetPacket((unsigned char *)kbuf, count, NULL, 0, 0);
    mutex_unlock(&rf->data_pipe.write_mutex);

    kfree(kbuf);
    return err;
}

static long mc13192_unlocked_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    int err;
    struct mc13192 *rf = filp->private_data;
    mutex_lock(&rf->ioctl_mutex);
    err = mc13192_ioctl(filp, cmd, arg);
    mutex_unlock(&rf->ioctl_mutex);

    return err;
}

const struct file_operations mc13192_data_fops = {
    .owner      = THIS_MODULE,
    .open       = mc13192_data_open,
    .release    = mc13192_release,
    .read       = mc13192_data_read,
    .write      = mc13192_data_write,
    .unlocked_ioctl = mc13192_unlocked_ioctl,
};
const struct file_operations mc13192_ctl_fops = {
    .owner      = THIS_MODULE,
    .open       = mc13192_ctl_open,
    .release    = mc13192_release,
    .read       = mc13192_ctl_read,
};
const struct file_operations mc13192_diag_fops = {
    .owner      = THIS_MODULE,
    .open       = mc13192_diag_open,
    .release    = mc13192_release,
    .read       = mc13192_diag_read,
};

#ifndef MC_USE_CDEV
static struct miscdevice mc13192_dev = {
    .minor      = MISC_DYNAMIC_MINOR,
    .name       = DRIVER_NAME,
//  .nodename   = "pds",
    .fops       = &mc13192_fops
};
#endif
//static struct cdev mc13192_dev = {
//    //FIXME: Not even remotely the correct minor number, arbitrarily chosen.
//    .minor      = MWAVE_MINOR,
//    .name       = DRIVER_NAME,
////  .nodename   = "pds",
//    .fops       = &mc13192_fops
//};

void mc13192_timer_function(unsigned long data);
inline static void mc13192_start_timer(struct mc13192 *rf)
{
    rf->timer_enabled = true;
    mod_timer(&rf->my_timer, jiffies + msecs_to_jiffies(RFREC_TIMER_RATE_MSEC));
}

void mc13192_timer_function(unsigned long data)
{
    struct mc13192 *rf = (struct mc13192*)data;

    // Call the platform agnostic timer function to do the work.
    RFReceive_Timer_HISR_Function();

    /* Need to restart timer for next expiration. Can only re-enable
     * if we haven't been disabled. */
    if(rf->timer_enabled) {
        mc13192_start_timer(rf);
    }
}

UINT16 RFReceive_DisableTimerInterrupt (void)
{
    struct mc13192 *rf = spi_get_drvdata(hprf_spi);
    rf->timer_enabled = false;
    del_timer_sync(&rf->my_timer);
    return SUCCESS;
}
UINT16 RFReceive_EnableTimerInterrupt (void)
{
    struct mc13192 *rf = spi_get_drvdata(hprf_spi);
    mc13192_start_timer(rf);
    return SUCCESS;
}

void RFReceive_TimerInitialize (void)
{
    struct spi_device *spi = to_spi_device(&hprf_spi->dev);
    struct mc13192 *rf = spi_get_drvdata(spi);
    init_timer(&rf->my_timer);
    rf->my_timer.data = (unsigned long)(rf);
    rf->my_timer.function = mc13192_timer_function;
}
UINT16 RFReceive_GetTimerValue (void) { return jiffies; }
void RFReceive_TimerPowerDown(void) {}
void RFReceive_TimerPowerUp(void) {}
static void mc13192_worker(struct work_struct *work);
static struct workqueue_struct *mc13192_wq;

void mc13192_worker(struct work_struct *work)
{
//  pr_info("============ In worker ============\n");
    struct mc13192 *rf = container_of(work, struct mc13192, work);
    MC13192_HISR();
    enable_irq(rf->spi->irq);
    return;
}

static irqreturn_t mc13192_irq_wq(int irq, void *handle)
{
    struct mc13192 *rf = (struct mc13192*) handle;
    disable_irq_nosync(irq);
//    pr_info("============ mc13192_irq ============ irq = %i, rf = %p mc13192_wq = %p\n",
//          irq, rf, mc13192_wq);
    queue_work(mc13192_wq, &rf->work);
    return IRQ_HANDLED;
}

/**
 * ISR, work queues because we need to sleep.
 */
static irqreturn_t mc13192_irq_thread(int irq, void *handle)
{
    MC13192_HISR();
    enable_irq(irq);
    return IRQ_HANDLED;
}

static irqreturn_t mc13192_irq(int irq, void *handle)
{
    disable_irq_nosync(irq);
    return IRQ_WAKE_THREAD;
}

void MC13192_IRQ_Disable(void)
{
    disable_irq(hprf_spi->irq);
}
void MC13192_IRQ_Enable(void)
{
    enable_irq(hprf_spi->irq);
}
void IRQPinEnable(void)
{
    //MC13192_IRQ_Enable();
}
void IRQACK(void) {}
void irq_init(struct mc13192 *rf)
{
	int err;
	struct spi_device *spi = rf->spi;
	dev_info(&spi->dev, "spi = %p hprf_spi = %p\n", spi, hprf_spi);
	err = request_irq(spi->irq, mc13192_irq_wq, IRQF_TRIGGER_FALLING, spi->dev.driver->name, rf);
	//err = devm_request_threaded_irq(spi->irq, NULL, mc13192_irq, IRQF_TRIGGER_FALLING, spi->dev.driver->name, rf);
    //err = request_threaded_irq(spi->irq, mc13192_irq, mc13192_irq_thread, IRQF_TRIGGER_FALLING, spi->dev.driver->name, rf);
	//err = request_threaded_irq(spi->irq, NULL, mc13192_irq_thread, IRQF_TRIGGER_FALLING, spi->dev.driver->name, rf);
	if (err)
	{
		dev_err(&hprf_spi->dev, "%s() - ERROR: Failed IRQ request: %d\n", __FUNCTION__, err);
	}
}
void SMAC_IRQInit(void)
{
    /**
     * Horrible kludge. Take our hprf_spi global pointer, and get the *real*
     * spi_device from its dev pointer.
     */
	struct spi_device *spi = to_spi_device(&hprf_spi->dev);
	struct mc13192 *rf = spi_get_drvdata(spi);
	irq_init(rf);
}

static void test_read(struct spi_device *spi)
{
	int i;
	size_t num_reg = ARRAY_SIZE(reg_arr);
	for(i=0; i < num_reg; ++i) {
		u8 addr = reg_arr[i];
		ssize_t res =  spi_w8r16(spi, addr | (0x80));
		dev_info(&spi->dev, "spi [0x%02x] = 0x%04x\n", addr, res > 0 ? ntohs(res) : res);
	}
}

typedef struct
{
	u8 addr;
	u16 value;
} reg_value;

reg_value init_values[] = {
		{ 0x31, 0xa0c0 },
		{ 0x34, 0xfec6 }
};

static void write_regs(struct spi_device *spi, reg_value regs[], int num_regs)
{
	int i;
	uint8_t buf[3];
	for(i = 0; i < num_regs; ++i)
	{
		buf[0] = regs[i].addr;
		buf[1] = (regs[i].value & 0xff00) >> 8;
		buf[2] = (regs[i].value & 0x00ff);
		spi_write(spi, buf, 3);
	}
}

static void dump_regs(struct spi_device *spi)
{
	int i;
	const int num_regs = 0x34;
	uint16_t buf[num_regs];
	size_t byteSize = ARRAY_SIZE(buf)*2;
	memset(buf, 0, byteSize);
	for(i=0; i < num_regs; ++i) {
		ssize_t res = spi_w8r16(spi, i | (0x80));
		if(res > 0) {
			buf[i] = ntohs(res);
		}
	}
	print_hex_dump(KERN_INFO, "hprf: ", DUMP_PREFIX_OFFSET,
			16, 2, buf, byteSize, false);
}

void init_structs(void)
{
	Rf_Smac_802_PHYParams.CCAEnergyThreshold = 10;
	Rf_Smac_802_PHYParams.PowerCompOffset = 5;
    memcpy(Rf_Smac_802_PHYParams.deviceExtendedAddress, bogus_uid, UID_MAX_SIZE);
	Rf_Smac_802_PHYParams.outputPower = 10;

	//memcpy(Rf_Smac_802_PIB.coordExtendedAddress, uid, UID_MAX_SIZE);
	//Rf_Smac_802_PIB.coordShortAddress = 0xfeed;
    Rf_Smac_802_PIB.isRxOnWhenIdle = 1;
    Rf_Smac_802_PIB.logicalChannel = 12;
    //Rf_Smac_802_PIB.maxCSMABackoffs;
    //Rf_Smac_802_PIB.minBE;
	Rf_Smac_802_PIB.panId = 0xfeed;
	Rf_Smac_802_PIB.promiscuousMode = 0;
	Rf_Smac_802_PIB.rfDisabled  = 0;
	//Rf_Smac_802_PIB.shortAddress;
	//Rf_Smac_802_PIB.shouldPermitAssociate;

}

void init_defaults(struct mc13192 *rf)
{
/* TODO: Remove. These were test values. */
#if 0
    rf->lqiActive = FALSE;
    rf->localDevLqi = 255;
    rf->remoteLqi = 255;

    memcpy(&rf->rfparam.UID, bogus_uid, UID_MAX_SIZE);
    rf->rfparam.shortAddr = 0xfffe;
    rf->rfparam.addrMode = 3;
    //rf->rfparam.CCAEnergyDetectThr = 0x40;
    rf->rfparam.CCAEnergyDetectThr = 96;
    //rf->rfparam.RSSIPowerCompOffset = 0x74;
    rf->rfparam.RSSIPowerCompOffset = 0;
    rf->rfparam.channel = 14;
    rf->rfparam.retry = 3;
    rf->rfparam.resultcodetimeout = 2;
    rf->rfparam.heartBeat = 4;
    rf->rfparam.panID = 0xfeed;
    rf->rfparam.gatewayPanID = 0xfeed;
    //rf->rfparam.gatewayPanID = 0xffff;
    memcpy(&rf->rfparam.gatewayUID, bogus_gatewayUid, UID_MAX_SIZE);
    rf->rfparam.RFID = 0x06;
    strncpy(rf->bootIdMsg, bogus_bootid, MAX_BOOT_ID_MESSAGE);

    rf->rfparamlocal.TSID = 0x2839;
    //TODO: Crashes without this! Seems to be some sort of logic error.
    //If the UID can possibly be invalid, it should not cause the system
    //to crash if an attempt is made to acquire under those circumstances.
    rf->rfparamlocal.UID_Valid = UID_SIGN;

    rf->rfparam.RFTraceFlags = 0x7; // Turn on all debug flags.
#else
    memset(&rf->rfparam, 0, sizeof(rf->rfparam));
    rf->rfparam.panID = 0xFEED;
    rf->rfparam.shortAddr = 0xFFFE;
    rf->rfparam.addrMode = 0x03;
    rf->rfparam.CCAEnergyDetectThr = 0x96;
    rf->rfparam.RSSIPowerCompOffset = 0xB5;
    rf->rfparam.channel = 0x14;
    rf->rfparam.power = 0xFFFF;
    rf->rfparam.retry = 0x09;
    /* rf->rfparam.retrydelay already memset to 0 above. */
    rf->rfparam.resultcodetimeout = 0x0003;
    rf->rfparam.heartBeat = 0x04;
    /* rf->rfparam.link_retrydelay already memset to 0 above. */
    /* rf->rfparam.gatewayUID already memset to 0 above. */
    rf->rfparam.gatewayPanID = 0xFEED;
    rf->rfparam.RFID = 0x03;
    /* rf->rfparam.RFTraceFlags; */
    rf->rfparam.powerhilo = 0x01;
    rf->rfparam.LinkLQIThreshold = 0xB4;
    rf->rfparam.txPktretryonconnection = 0x03;
    rf->rfparam.PreLinkQueryPacketCnt = 0x01;
    rf->rfparam.powerBoost = 0x00;
#endif
}

BOOL readWriteEeprom(EEP_PARAMS *eep_param, BOOL doRead)
{
    BOOL retVal = false;
    ssize_t (*op)(char*, size_t, loff_t) = doRead ? m24m01_read : m24m01_write;
    const char *opstr = doRead ? "read" : "write";
    int result;
    result = (*op)(&eep_param->RfUID, UID_MAX_SIZE, CS_UC_ER_UID);
    if(result != UID_MAX_SIZE) {
        pr_err("%s: Failed to %s UID.\n", __FUNCTION__, opstr);
        goto cleanup;
    }
    result = (*op)(&eep_param->dwSignature, 4, CS_UL_SIGNATURE);
    if(result != 4) {
        pr_err("%s: Failed to %s signature.\n", __FUNCTION__, opstr);
        goto cleanup;
    }
    result = (*op)(&eep_param->dwVersion, 4, CS_UL_REVISIONID);
    if(result != 4) {
        pr_err("%s: Failed to %s version.\n", __FUNCTION__, opstr);
        goto cleanup;
    }
    result = (*op)(&eep_param->RfUIDSign, CS_SIZE_ER_UIDSIGN, CS_UL_ER_UIDSIGN);
    if(result != CS_SIZE_ER_UIDSIGN) {
        pr_err("%s: Failed to %s UID sign.\n", __FUNCTION__, opstr);
        goto cleanup;
    }
    result = (*op)(&eep_param->RfPowerTable, CS_SIZE_POWER_TABLE, CS_US_ER_POWER_TABLE);
    if(result != CS_SIZE_POWER_TABLE) {
        pr_err("%s: Failed to %s UID.\n", __FUNCTION__, opstr);
        goto cleanup;
    }
    retVal = true;
cleanup:
    return result;
}

void printEepParams(EEP_PARAMS *eep_param)
{
    pr_info("signature = %lx\n", eep_param->dwSignature);
    pr_info("version = %lx\n", eep_param->dwVersion);
    pr_info("uidsign = %lx\n", eep_param->RfUIDSign);
    pr_info("uid = %02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx\n",
            eep_param->RfUID[0], eep_param->RfUID[1], eep_param->RfUID[2], eep_param->RfUID[3],
            eep_param->RfUID[4], eep_param->RfUID[5], eep_param->RfUID[6], eep_param->RfUID[7]);
//    for(size_t i=0; i < UID_MAX_SIZE; ++i) {
//
//    }
    pr_info("signature = %lx\n", eep_param->dwSignature);
    pr_info("power table = %hx\n", eep_param->RfPowerTable);
}

BOOL verifyUserEepParams(EEP_PARAMS *eep_param)
{
    //RETAILMSG(1, (TEXT("VerifyUserEepParams: Found Sign: 0x%x, Ver: %d\r\n"), eep_param.dwSignature, eep_param.dwVersion));

    // First check the signature
    if (eep_param->dwSignature != EEP_SIGNATURE)
    {
        pr_info("EEP verifyUserEepParams: Using defaults.\n");
        // Empty flash. Restore default values if needed.

        // <<-- Add here if you have to restore any defaults.

        // At the end, put the signature and current version.
        eep_param->dwSignature = EEP_SIGNATURE;
        eep_param->dwVersion = EEP_VERSION;
        return FALSE;

    }
    //RETAILMSG(1, (TEXT("VerifyUserEepParams: Done.\r\n")));
    return TRUE;
}

void defaultErStruct(ER_RADIO_SETTINGS_STRUCT *erRadioSettings)
{
     erRadioSettings->ulER_PanID = 0xFEED;
     erRadioSettings->ulER_ShortAddr = 0xFFFE;
     erRadioSettings->ulER_AddrMode = 0x03;
     erRadioSettings->ulER_CCAEnergyDetectThr = 0x96;
     erRadioSettings->ulER_RSSIPowerCompOffset = 0xB5;
     erRadioSettings->ulER_Channel = 0x14;
     erRadioSettings->ulER_Power = 0xFFFF;
     erRadioSettings->ulER_Retry = 0x09;
     memset(&erRadioSettings->ulER_RetryDelay, 0, sizeof (erRadioSettings->ulER_RetryDelay));
     erRadioSettings->ulER_ResultcodeTimeout = 0x0003;
     erRadioSettings->ulER_Heartbeat = 0x04;
     memset(&erRadioSettings->ulER_LinkRetryDelay, 0, sizeof (erRadioSettings->ulER_LinkRetryDelay));
     memset(&erRadioSettings->ucER_GatewayUID, 0, UID_MAX_SIZE);
     erRadioSettings->ulER_GatewayPanID = 0xFEED;
     erRadioSettings->ulER_RFID = 0x03;
     erRadioSettings->ulER_RFTxDebug = 0x00;
     erRadioSettings->ulER_RFRxDebug = 0x00;
     erRadioSettings->ulER_RFGenericDebug = 0x00;
     erRadioSettings->ulER_PowerHiLo = 0x01;
     erRadioSettings->ulER_LinkLQIThreshold = 0xB4;
     erRadioSettings->ulER_TxPktretryonconnection = 0x03;
     erRadioSettings->ulER_PreLinkQueryPacketCnt = 0x01;
     erRadioSettings->ulER_PowerBoost = 0x00;
}

void copyErStructToRfParam(struct mc13192 *rf, ER_RADIO_SETTINGS_STRUCT *erRadioSettings)
{
    rf->rfparam.panID = (UINT16)erRadioSettings->ulER_PanID;
    rf->rfparam.shortAddr = (UINT16)erRadioSettings->ulER_ShortAddr;
    rf->rfparam.addrMode = (UINT8)erRadioSettings->ulER_AddrMode;
    rf->rfparam.CCAEnergyDetectThr = (UINT8)erRadioSettings->ulER_CCAEnergyDetectThr;
    rf->rfparam.RSSIPowerCompOffset = (UINT8)erRadioSettings->ulER_RSSIPowerCompOffset;
    rf->rfparam.channel = (UINT8)erRadioSettings->ulER_Channel;
    rf->rfparam.power = (UINT16)erRadioSettings->ulER_Power;
    rf->rfparam.retry = (UINT8)erRadioSettings->ulER_Retry;
    memcpy(&rf->rfparam.retrydelay, &erRadioSettings->ulER_RetryDelay, RF_MAX_NO_OF_RETRIES_DELAYS);
    rf->rfparam.resultcodetimeout = (UINT16)erRadioSettings->ulER_ResultcodeTimeout;
    rf->rfparam.heartBeat = (UINT8)erRadioSettings->ulER_Heartbeat;
    memcpy(&rf->rfparam.link_retrydelay, &erRadioSettings->ulER_LinkRetryDelay, RF_MAX_NO_OF_RETRIES_DELAYS);
    memcpy(&rf->rfparam.gatewayUID, &erRadioSettings->ucER_GatewayUID, UID_MAX_SIZE);
    rf->rfparam.gatewayPanID = (UINT16)erRadioSettings->ulER_GatewayPanID;
    rf->rfparam.RFID = (UINT8)erRadioSettings->ulER_RFID;
    rf->rfparam.RFTraceFlags |= erRadioSettings->ulER_RFTxDebug ? RF_RFTX_DEBUG_FLAG : 0;
    rf->rfparam.RFTraceFlags |= erRadioSettings->ulER_RFTxDebug ? RF_RFRX_DEBUG_FLAG : 0;
    rf->rfparam.RFTraceFlags |= erRadioSettings->ulER_RFTxDebug ? RF_GENERIC_DEBUG_FLAG : 0;
    rf->rfparam.powerhilo = (UINT8)erRadioSettings->ulER_PowerHiLo;
    rf->rfparam.LinkLQIThreshold = (UINT8)erRadioSettings->ulER_LinkLQIThreshold;
    rf->rfparam.txPktretryonconnection = (UINT8)erRadioSettings->ulER_TxPktretryonconnection;
    rf->rfparam.PreLinkQueryPacketCnt = (UINT8)erRadioSettings->ulER_PreLinkQueryPacketCnt;
    rf->rfparam.powerBoost = (UINT8)erRadioSettings->ulER_PowerBoost;
}

void copyParamsIntoErStruct(ER_RADIO_SETTINGS_STRUCT *erRadioSettings, struct mc13192 *rf)
{
    pr_err("\nERROR HPRF Init: VerifyUserEepParams Fail! using defaults\n");
 }

static void loadDStoreParams(struct mc13192 *rf)
{
    BOOL retVal;
    int bytesWritten;
    EEP_PARAMS eep_param;
    ER_RADIO_SETTINGS_STRUCT erRadioSettings;

    // Clear the structure
    memset(&eep_param, 0, sizeof (EEP_PARAMS));
    memset(&rf->rfparam, 0, sizeof (RF_PARAM));
    retVal = readWriteEeprom(&eep_param, 1);

    printEepParams(&eep_param);

    defaultErStruct(&erRadioSettings);
    if(!verifyUserEepParams(&eep_param)) {
        // Init defaults in eeprom.
        dev_err(&rf->spi->dev, "VerifyUserEepParams Fail! using defaults.\n");
        copyErStructToRfParam(rf, &erRadioSettings);
        bytesWritten = m24m01_write(&erRadioSettings, sizeof(ER_RADIO_SETTINGS_STRUCT), DS_ER_RADIO_SETTINGS);
        if(bytesWritten != sizeof(ER_RADIO_SETTINGS_STRUCT)) {
            pr_err("HPRF Write DataStore Error.\n");
        }
        return;
    }
    if(eep_param.RfUIDSign == UID_SIGN) {
        // Save that we have a valid UID
        rf->rfparamlocal.UID_Valid = UID_SIGN;
        memcpy(&rf->rfparam.UID[0], eep_param.RfUID, UID_MAX_SIZE);
    }
    else {
        dev_err(&rf->spi->dev, "HPRF InitFound invalid signature in EEPROM. Restoring defaults.\n");
        bytesWritten = m24m01_write(&erRadioSettings, sizeof(ER_RADIO_SETTINGS_STRUCT), DS_ER_RADIO_SETTINGS);
        if(bytesWritten != sizeof(ER_RADIO_SETTINGS_STRUCT)) {
            pr_err("HPRF Write DataStore Error.\n");
        }
        rf->rfparamlocal.UID_Valid = 0;
        memset(&rf->rfparam.UID[0], 0, UID_MAX_SIZE);
    }
    bytesWritten = m24m01_read(&erRadioSettings, sizeof(ER_RADIO_SETTINGS_STRUCT), DS_ER_RADIO_SETTINGS);
    copyErStructToRfParam(rf, &erRadioSettings);
}

#ifdef MC_USE_CDEV

static int mc13192_register_one_device(struct class *one_class,
        struct cdev *one_cdev, struct file_operations const *fops, const char *name,
        int dev_index, struct device **one_dev)
{
    int err = 0;
    dev_t devno = MKDEV(MAJOR(mc13192_devno), MINOR(mc13192_devno)+dev_index);
    cdev_init(one_cdev, fops);
    one_cdev->owner = THIS_MODULE;
    one_cdev->ops = fops;
    err = cdev_add(one_cdev, devno, 1);
    if(err) {
        pr_err("Failed to add cdev device. Error = %i\n", err);
        return err;
    }

    (*one_dev) = device_create(one_class, NULL, devno, NULL, name);
    if (IS_ERR(one_dev)) {
        pr_err("Failed to create character device\n");
        err = PTR_ERR(one_dev);
    }
    return err;
}

static int mc13192_create_class(struct mc13192 *rf)
{
    int err = 0;
    if(!mc13192_class) {
        mc13192_class = class_create(THIS_MODULE, "mc13192");
        if (IS_ERR(mc13192_class)) {
            pr_err("Failed to create mc13192 device class\n");
            err = PTR_ERR(mc13192_class);
        }
    }
    return err;

    //TODO: Move this where it might actually do some good.
cleanup:
    class_destroy(mc13192_class);
    mc13192_class = 0;
    return err;
}
#else
static int mc13192_register_devices(struct mc13192 *rf)
{
    int err;
    err = misc_register(&mc13192_dev);
    return err;
}
#endif

static int mc13192_init_pipe(struct mc13192_pipe_dev *pipe, int index,
        struct file_operations const *fops, const char *name)
{
    int err;
    spin_lock_init(&pipe->slock);
    mutex_init(&pipe->read_mutex);
    mutex_init(&pipe->write_mutex);
    init_waitqueue_head(&pipe->wq);
    err = kfifo_alloc(&pipe->fifo, USERSPACE_QUEUE_SIZE, GFP_KERNEL);
    if(err) {
        pr_err("Failed to allocate memory for pipe\n");
        return err;
    }

    err = mc13192_register_one_device(mc13192_class, &pipe->cdev,
            fops, name, index, &pipe->dev);
    return err;
}

static int mc13192_init_pipes(struct mc13192 *rf)
{
    mc13192_init_pipe(&rf->data_pipe,  0, &mc13192_data_fops, "mc13192_data");
    mc13192_init_pipe(&rf->ctl_pipe,   1, &mc13192_ctl_fops, "mc13192_ctl");
    mc13192_init_pipe(&rf->diag_pipe, 2, &mc13192_diag_fops, "mc13192_debug");
    return 0;
}

static int mc13192_destroy_pipes(struct mc13192 *rf)
{
    //TODO: Actually implement this?
    return 0;
}

static int __devinit mc13192_probe(struct spi_device *spi)
{
	struct mc13192 *rf;
	struct mc13192_platform_data *pdata;
	int err = 0;

	/* TODO: FIXME: HACK: Generalized dm_spi interface not implemented yet.
	 * See dm_spi.c and smac_mcu_spi_config_dmSpecific. */
	hprf_spi = spi;

	/* assuming the driver requires board-specific data: */
	/*TODO: Not setting the platform_data anywhere. */
	pdata = (struct mc13192_platform_data*)&spi->dev.platform_data;
	if (!pdata) {
		err = -ENODEV;
		goto probe_return;
	}
	/* get memory for driver's per-chip state */
	rf = kzalloc(sizeof(*rf), GFP_KERNEL);
	if (!rf) {
		err = -ENOMEM;
		goto probe_return;
	}

	mutex_init(&rf->ioctl_mutex);

	spi_set_drvdata(spi, rf);
	rf->spi = spi;

	INIT_WORK(&rf->work, mc13192_worker);

	err = sysfs_create_group(&spi->dev.kobj, &mc13192_attribute_group);
	if (err) {
		dev_err(&spi->dev, "ERROR: sysfs_create_group() failed: %d\n", err);
		goto error_free_fifo;
	}
    err = mc13192_create_class(rf);
	if(err) {
	    dev_err(&spi->dev, "Failed to create device class - %i\n", err);
	    goto error_free_sysfs;
	}
	err = mc13192_init_pipes(rf);
    if(err) {
        dev_err(&spi->dev, "Failed to initialize pipes - %i\n", err);
        goto error_free_sysfs;
    }

    init_structs();
//	write_regs(spi, init_values, ARRAY_SIZE(init_values));
	/* Inits critical section mutex. */
	OsStart();
	loadDStoreParams(rf);
//    init_defaults(rf);
    //RF_Intfc_Init(12, 5, &uid[0], 96, 0);
//    RF_Intfc_Init(rf->rfparam.channel,
//                  rf->rfparam.power,
//                  &rf->rfparam.UID[0],
//                  rf->rfparam.CCAEnergyDetectThr,
//                  rf->rfparam.powerBoost);
//    Init_Smac_802_15_4_Shim();

    g_startupEventGroup = OsCreateEventGroup();
    rf->lqiEvent = OsCreateEventGroup();
    RFIntRFApp.pF_InitRFApplication();
    rf->lqiThread = kthread_run(rf_lqi_thread, 0, "HPRFLQI");

	//irq_init(rf);
	//char buf[] = {'\0', '\0', '\0' };
	//int res = spi_write(spi, buf, ARRAY_SIZE(buf)); /* soft reset */

	//dump_regs(spi);
	goto probe_return;
error_free_sysfs:

#ifdef MC_USE_CDEV
    //TODO: destroy class.
#else
    misc_deregister(&mc13192_dev);
#endif

//	if (input_device)
	{
//		input_free_device(input_device);
//		destroy_workqueue(mc13192_wq);
//		ts->ctrl.ts_dev = NULL;
	}
error_free_fifo:
    mc13192_destroy_pipes(rf);
    mutex_destroy(&rf->ioctl_mutex);
error_free_data:
	kfree(rf);
probe_return:
	return err;
}

static int __devexit mc13192_remove(struct spi_device *spi)
{
	return 0;
}

#define HP_ECSPI_1_CLK        IMX_GPIO_NR(3,16)
#define HP_ECSPI_1_MOSI       IMX_GPIO_NR(3,18)
#define HP_ECSPI_1_MISO       IMX_GPIO_NR(3,17)
#define HP_ECSPI_1_CS1        IMX_GPIO_NR(3,19)

static iomux_v3_cfg_t suspend_pin_cfg[] =
{
    MX53_PAD_EIM_D16__GPIO3_16,
    MX53_PAD_EIM_D17__GPIO3_17,
    MX53_PAD_EIM_D18__GPIO3_18,
    MX53_PAD_EIM_D19__GPIO3_19
};

static iomux_v3_cfg_t resume_pin_cfg[] =
{
    MX53_PAD_EIM_D16__ECSPI1_SCLK,
    MX53_PAD_EIM_D17__ECSPI1_MISO,
    MX53_PAD_EIM_D18__ECSPI1_MOSI,
    MX53_PAD_EIM_D19__GPIO3_19
};

static int mc13192_suspend(struct spi_device *spi, pm_message_t mesg)
{
    int err;
    struct mc13192 *rf = spi_get_drvdata(spi);
    dev_info(&spi->dev, "powering down\n");
    rf->hack_isSleeping = 1;
    disable_irq(spi->irq);
    flush_workqueue(mc13192_wq);
    RFIntRFApp.pF_PowerDown();

    // Power down the hardware
    gpio_set_value(HPRF_GPIONUM_RADIO_ON, 0);

    // Hold in reset
    gpio_set_value(HPRF_GPIONUM_N_RESET, 0);

    // As per HW engineers request, all input's into iMX53 should be set to a 360K internal
    // pull down during power down mode.
    // Also, all the outputs should be driven low, namely
    // RF_TXEN, RF_RXEN, RF_RESET_N, RF_PA_EN, RF_RXTX_EN, RF_ATTN_N
    /* TODO: Configure PATA_DATA14,13,12,15, (ID0,ID1,ID2,INT_N)*/
    err = mxc_iomux_v3_setup_multiple_pads(suspend_pin_cfg, ARRAY_SIZE(suspend_pin_cfg));
    if(err) {
        dev_err(&spi->dev, "failed to set pins\n");
    }
    gpio_direction_output(HPRF_GPIONUM_FCC_ID01, 0);
    gpio_direction_output(HPRF_GPIONUM_FCC_ID02, 0);
    gpio_direction_output(HPRF_GPIONUM_FCC_ID03, 0);
    gpio_direction_output(HPRF_GPIONUM_N_IRQ, 0);

    gpio_set_value(HPRF_GPIONUM_PAEN, 0);
    gpio_set_value(HPRF_GPIONUM_N_RXEN, 0);
    if(crest_get_prodID() != TST900)		// JAS TX_EN is not used on 900
		gpio_set_value(HPRF_GPIONUM_N_TX_EN, 0);
    gpio_set_value(HPRF_GPIONUM_RXTXEN, 0);
    gpio_set_value(HPRF_GPIONUM_N_ATTN, 0);

    gpio_direction_output(HP_ECSPI_1_CLK, 0);
    gpio_direction_output(HP_ECSPI_1_MOSI, 0);
    gpio_direction_output(HP_ECSPI_1_MISO, 0);
    gpio_direction_output(HP_ECSPI_1_CS1, 0);
	return 0;
}

static int mc13192_resume(struct spi_device *spi)
{
    int err;
    struct mc13192 *rf = spi_get_drvdata(spi);
    dev_info(&spi->dev, "powering up\n");

    /* TODO: Reinitialize pins after power up */
    err = mxc_iomux_v3_setup_multiple_pads(resume_pin_cfg, ARRAY_SIZE(suspend_pin_cfg));
    if(err) {
        dev_err(&spi->dev, "failed to set pins\n");
    }
    gpio_direction_input(HPRF_GPIONUM_FCC_ID01);
    gpio_direction_input(HPRF_GPIONUM_FCC_ID02);
    gpio_direction_input(HPRF_GPIONUM_FCC_ID03);
    gpio_direction_input(HPRF_GPIONUM_N_IRQ);

    gpio_set_value(HPRF_GPIONUM_N_RESET, 1);
    gpio_set_value(HPRF_GPIONUM_RADIO_ON, 1);

    RFIntRFApp.pF_PowerUp();
    enable_irq(spi->irq);
    rf->hack_isSleeping = 0;
    dev_info(&spi->dev, "powered up\n");
    if (rf->rfparamlocal.UID_Valid == UID_SIGN && Smac_PhyInitialized)
    {
        //If we want to avoid sending connect which causes update request for unknown wakeup
        //For example, see \PLATFORM\MAINSTONEIII\SRC\DRIVERS\PWRBUTTON\pwrbutton.c
        //KernelIoControl(IOCTL_HAL_GET_WAKE_SOURCE, NULL, 0, &wakeSrc, sizeof(wakeSrc), &bytesRet);

        // Send connect message here itself. Other ways is to put it in SetPowerState when the
        // state changes from D3 to below. But with that it is noticed that the first key presses
        // are sometimes missed sending to control system.
        RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();
    }
	return 0;
}

static struct spi_driver mc13192_driver = {
	.driver = {
		.name       = DRIVER_NAME,
		.owner      = THIS_MODULE,
	},

	.probe      = mc13192_probe,
	.remove     = __devexit_p(mc13192_remove),
	.suspend    = mc13192_suspend,
	.resume     = mc13192_resume,
};

static int mc13192_init(void)
{
    int err;
#ifdef MC_USE_CDEV
    err = alloc_chrdev_region(&mc13192_devno, 0, MC13192_MAX_DEVICES, "mc13192");
    if(err) {
        pr_err("mc13192: Could not allocate device numbers.\n");
        return err;
    }
    else {
        pr_info("Allocated %u devices starting at (%u,%u)\n",
                MC13192_MAX_DEVICES, MAJOR(mc13192_devno), MINOR(mc13192_devno));
    }
#endif
    mc13192_wq = create_singlethread_workqueue("mc13192_wq");
//	mc13192_wq = alloc_workqueue("mc13192_wq", WQ_MEM_RECLAIM | WQ_HIGHPRI, 1);
    if(!mc13192_wq) {
    	pr_err("mc13192: Could not create workqueue.\n");
#ifdef MC_USE_CDEV
        unregister_chrdev_region(mc13192_devno, MC13192_MAX_DEVICES);
#endif
        return -ENOMEM;
    }
    return spi_register_driver(&mc13192_driver);
}

static void mc13192_exit(void)
{
#ifdef MC_USE_CDEV
	unregister_chrdev_region(mc13192_devno, MC13192_MAX_DEVICES);
#endif
    if (mc13192_wq)
    {
        destroy_workqueue(mc13192_wq);
    }
	spi_unregister_driver(&mc13192_driver);
}

MODULE_LICENSE("GPL");

late_initcall(mc13192_init);
module_exit(mc13192_exit);
